#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import sys
import os
import ecore
import dbus
try:
    from e_dbus import DBusEcoreMainLoop
    DBusEcoreMainLoop(set_as_default=True)
except Exception:
    import dbus.ecore

import logging
log = logging.getLogger("plugins.canola-core.maemo.system")

from terra.core.terra_object import TerraObject
from terra.core.model import ModelStatus
from terra.core.controller import Controller
from terra.ui.window import StatusIcon

pjoin = os.path.join

class BatteryStatus(ModelStatus):
    terra_type = "Model/Status/Battery"

    (BATTERY_CHARGING_OFF,
     BATTERY_CHARGING_ON) = xrange(2)

    def __init__(self):
        ModelStatus.__init__(self, "Battery")
        self.status = (self.BATTERY_CHARGING_OFF, 0.0)
        self._setup()

    def _setup(self):
        sbus = dbus.SystemBus()

        sigs_template = [
            (self._update_battery_level, "battery_state_changed"),
            (self._battery_level_low, "battery_low"),
            (self._battery_level_low, "battery_empty"),
            (self._battery_level_full, "battery_full"),
            (self._battery_charging_off, "charger_charging_off"),
            (self._battery_charging_on, "charger_charging_on"),
            ]

        add_handler = sbus.add_signal_receiver
        for meth, sname in sigs_template:
            add_handler(meth, signal_name=sname,
                        dbus_interface="com.nokia.bme.signal",
                        path="/com/nokia/bme/signal")

        # send request for battery information
        # FIXME: UGLY UGLY HACK because of old dbus/dbus-python!
        os.system("dbus-send --system /com/nokia/bme/request "
                  "com.nokia.bme.request.status_info_req")

    def _update_battery_level(self, level):
        # Battery levels are 0 to 4. We translate this to 0.0 to 1.0 scale, but
        # reserving value 1.0 for battery full.
        p = level * 0.25
        if p == 1.0:
            p = 0.99
        self.status = (self.status[0], p)

    def _battery_level_low(self):
        self.status = (self.BATTERY_CHARGING_OFF, 0.0)

    def _battery_level_full(self):
        log.debug("Battery is full.")
        self.status = (self.BATTERY_CHARGING_ON, 1.0)

    def _battery_charging_off(self):
        # Here we hack the status so is never 1.0, which is reserved for
        # fully charged.
        p = self.status[1]
        if p == 1.0:
            p = 0.99
        self.status = (self.BATTERY_CHARGING_OFF, p)

    def _battery_charging_on(self):
        self.status = (self.BATTERY_CHARGING_ON, self.status[1])


class BatteryStatusController(Controller):
    terra_type = "Controller/Status/Battery"

    MSG_ID = 1

    def __init__(self, model, canvas, parent):
        Controller.__init__(self, model, canvas, parent)
        self.view = StatusIcon(parent.view, "battery")

        self.model.add_listener(self._update_battery)
        self.view.callback_theme_changed = self._theme_changed

    def _theme_changed(self, *ignored):
        self._update_battery(self.model)

    def _update_battery(self, model):
        log.debug("Setting battery status to (%d, %f).", model.status[0],
                  model.status[1])
        if model.status[0] == model.BATTERY_CHARGING_OFF:
            signal = "state,charging_off"
        else:
            signal = "state,charging_on"

        self.view.signal_emit(signal, "canola")
        self.view.message_send(self.MSG_ID, model.status[1])

    def delete(self):
        self.view.callback_theme_changed = None
        self.model.remove_listener(self._update_battery)
        self.view.delete()


class NetworkStatus(ModelStatus):
    terra_type = "Model/Status/Network"

    def __init__(self):
        ModelStatus.__init__(self, "Network")
        self.status = 0.0
        self._setup()

    def _setup(self):
        sbus = dbus.SystemBus()
        icd = sbus.get_object("com.nokia.icd",
                              "/com/nokia/icd",
                              introspect=False)
        self._icd_iface = dbus.Interface(icd, "com.nokia.icd")

        self._req_conn_stats()
        self._timer = ecore.timer_add(2.5, self._req_conn_stats)

    def _stats_reply(self, *stats):
        def signed(value):
            if value > sys.maxint:
                value = value - 2L * (sys.maxint + 1)
            assert -sys.maxint-1 <= value <= sys.maxint
            return value

        try:
            signal = signed(stats[2])
        except:
            self.status = 0.0
            return

        if signal > -60:
            signal = 1.0
        elif signal > -80:
            signal = 0.6
        else:
            signal = 0.3

        self.status = signal

    def _stats_error(self, *ignored):
        self.status = 0.0

    def _req_conn_stats(self):
        self._icd_iface.get_statistics(reply_handler=self._stats_reply,
                                       error_handler=self._stats_error)
        return True


class NetworkStatusController(Controller):
    terra_type = "Controller/Status/Network"

    MSG_ID = 1

    def __init__(self, model, canvas, parent):
        Controller.__init__(self, model, canvas, parent)
        self.view = StatusIcon(parent.view, "connection")
        self.model.add_listener(self._update_network)
        self.view.callback_theme_changed = self._theme_changed

    def _theme_changed(self, *ignored):
        self._update_network(self.model)

    def _update_network(self, model):
        self.view.message_send(self.MSG_ID, model.status)

    def delete(self):
        self.view.callback_theme_changed = None
        self.model.remove_listener(self._update_network)
        self.view.delete()


class ScreenPowerSave(TerraObject):
    terra_type = "ScreenPowerSave"

    def __init__(self):
        TerraObject.__init__(self)
        self._setup()
        self.locked = False

    def _setup(self):
        sbus = dbus.SystemBus()
        mce = sbus.get_object("com.nokia.mce",
                              "/com/nokia/mce/request",
                              introspect=False)
        self._mce_iface = dbus.Interface(mce, "com.nokia.mce.request")
        self._timer = None

    def _do_nothing(self, *ignored):
        pass

    def _state_on(self, *ignored):
        self._mce_iface.req_display_blanking_pause(
            reply_handler=self._do_nothing,
            error_handler=self._do_nothing)

    def _do_disable(self):
        log.debug("Asking system to keep the screen light on.")
        self._mce_iface.req_display_state_on(reply_handler=self._state_on,
                                             error_handler=self._do_nothing)
        return True

    def _enabled_get(self):
        return self._timer is None

    def _enabled_set(self, value):
        if self.locked:
            return

        if value:
            if self._timer is None:
                return
            self._timer.stop()
            self._timer = None
        else:
            if self._timer is not None:
                return
            self._do_disable()
            self._timer = ecore.timer_add(45.0, self._do_disable)

    enabled = property(_enabled_get, _enabled_set)


class SystemProperties(TerraObject):
    terra_type = "SystemProperties"

    HOME = os.environ["HOME"]
    MEDIA_BASE = pjoin(HOME, "MyDocs")
    DEFAULT_DIR_DOCS = pjoin(MEDIA_BASE, ".documents")
    DEFAULT_DIR_GAMES = None
    DEFAULT_DIR_IMAGES = pjoin(MEDIA_BASE, ".images")
    DEFAULT_DIR_AUDIO = pjoin(MEDIA_BASE, ".sounds")
    DEFAULT_DIR_VIDEO = pjoin(MEDIA_BASE, ".videos")

    storage_cards = ["/media/mmc1", "/media/mmc2"]

    DEFAULT_DIRS = {
        "photo": [DEFAULT_DIR_IMAGES] + storage_cards,
        "audio": [DEFAULT_DIR_AUDIO] + storage_cards,
        "video": [DEFAULT_DIR_VIDEO] + storage_cards}

    path_alias = {
        "/": "Root",
        "/media/mmc1": "Removable card",
        "/media/mmc2": "Internal card",
        HOME: "Home",
        DEFAULT_DIR_DOCS: "Documents",
        DEFAULT_DIR_IMAGES: "Images",
        DEFAULT_DIR_AUDIO: "Audio clips",
        DEFAULT_DIR_VIDEO: "Video clips",
        MEDIA_BASE: "My home directory"
        }

    def download_dirs_get(self):
        dd = []
        for dir in self.storage_cards + [self.MEDIA_BASE]:
            if dir in self.path_alias:
                dd.append((os.path.join(dir, "My downloads"),
                           self.path_alias[dir]))
        return dd

    def path_alias_get(self, path):
        p = os.path.normpath(os.path.expanduser(path))
        return self.path_alias.get(p, "")

    def basemountdir(self, path):
        volatile_path = None

        for known_mount in self.storage_cards:
            if path.startswith(known_mount):
                volatile_path = known_mount
                break

        return volatile_path

    def prepare_write_path(self, path, name, min_size=-1):
        errmsg = None

        if not self.is_mounted(path):
            basedir = self.basemountdir(path)
            alias = self.path_alias_get(basedir)

            if alias != "":
                errmsg = alias + " is not available."
            else:
                errmsg = name.capitalize() + " is on a non-available device."
            raise Exception(errmsg)

        if not os.path.exists(path):
            try:
                os.makedirs(path)
            except OSError, e:
                errmsg = e.strerror + ": '" +  e.filename + "'."
                raise Exception(errmsg)

        if min_size != -1 and self.bytes_available(path) < min_size:
            errmsg = "Not enough space left in " + name.lower() + "."
            raise Exception(errmsg)

    def is_mounted(self, path):
        volatile_path = self.basemountdir(path)

        if volatile_path is not None:
            memcards_paths = self.mounted_storage_cards_get()
            if volatile_path not in memcards_paths:
                return False

        return True

    def mounted_storage_cards_get(self):
        memcards = []
        try:
            mounts = open("/proc/mounts").read()
            for memcard_dir in self.storage_cards:
                if memcard_dir in mounts:
                    memcards.insert(0, memcard_dir)
        except:
            pass

        return memcards

    def bytes_available(self, path):
        if os.path.isfile(path):
            path = os.path.dirname(path)

        return int(os.statvfs(path)[4] * os.statvfs(path)[1])
