#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import os
import edje.decorators
import logging
import evas
import ecore

from terra.ui.base import EdjeWidget
from terra.core.manager import Manager
from terra.core.model import ModelFolder
from efl_utils.scroller import TextScrollerLeft

mger = Manager()
RemotePlayer = mger.get_class("Controller/RemotePlayer")
PlayerController = mger.get_class("Controller/Media")
AudioPlayerOptionsModel = mger.get_class("Model/Options/Folder/Player/Audio")
AudioOnTheGoOptionsMF = \
    mger.get_class("Model/Options/Folder/Media/Audio/Local/Playlist/OnTheGo")
DirectoryModel = mger.get_class("Model/Folder/Directory")
ThrobberDialog = mger.get_class("Model/ThrobberDialog")
AudioPlaylistOTGHook = mger.get_class("Hook/OnTheGo/AudioPlaylist")
EntryModal = mger.get_class("Widget/Settings/EntryModal")
DeletableRowRendererWidget = mger.get_class("Widget/DeletableRowRenderer")
BaseDeletableRowRenderer = mger.get_class("Widget/BaseDeletableRowRenderer")
ActionButton = mger.get_class("Widget/ActionButton")


log = logging.getLogger("plugins.canola-core.audio.ui")


class AudioPlayerScreen(EdjeWidget):
    def __init__(self, model, repeat, shuffle, parent, theme=None):
        EdjeWidget.__init__(self, parent.evas, "player_audio", parent, theme)
        self.callback_set_rating = None
        self.parent_view = parent
        self.cover = self.evas.FilledImage()
        self.title = None
        self.setup_widgets(repeat, shuffle)

    def setup_widgets(self, repeat=None, shuffle=None):
        self.parent_view.signal_emit("media_area,audio", "")
        self.pl = self.parent_view.part_swallow_get("player_view/playlist_status")
        self.w_cover = self.part_swallow_get("player_audio/album_cover")
        self.media_info = self.part_swallow_get("player_audio/media_info")
        self.rating_bar = self.media_info.part_swallow_get("rating_area")
        if self.title is None:
            self.title = TextScrollerLeft(canvas=self.parent_view.evas,
                                          period=0, n_scroll=0, text="",
                                          font=("Nokia Sans:style=Bold", 30),
                                          color="#ffffff")
        self.media_info.part_swallow("track_title", self.title)
        self.signal_callback_add("mouse,down,1",
                                 "player_audio/media_info:track_title",
                                 self.scroll_start)

        self.w_cover.part_swallow("album_cover", self.cover)
        self.parent_view.part_swallow("player_view/audio_throbber",
                                      self.parent_view.throbber)
        if repeat is None:
            repeat = self.repeat
        if shuffle is None:
            shuffle = self.shuffle
        self._setup_playlist_status(repeat, shuffle)

    def setup_model(self, model=None):
        if model:
            self.model = model
            self._setup_cover()
            self._setup_rating()
            self._setup_audio_labels()

    def _setup_audio_labels(self):
        if self.model.title:
            self.title.text = self.model.title
        else:
            self.title.text = ""

        if self.model.artist:
            artist = self.model.artist
            self.media_info.part_text_set("track_artist", artist)
        else:
            self.media_info.part_text_set("track_artist", "")

        if self.model.album:
            album = self.model.album
            self.media_info.part_text_set("track_album", album)
        else:
            self.media_info.part_text_set("track_album", "")

    def scroll_start(self, *ignored):
        if self.title.text1.size[0] > self.title.clipper.size[0]:
            self.title.start()

    def scroll_stop(self, *ignored):
        self.title.stop()
        self.title._setup_start_pos()

    def _setup_cover(self):
        cover = self.model.cover
        if cover and os.path.exists(cover):
            try:
                self.cover.file_set(cover)
            except Exception, e:
                log.error("could not setup cover %s (error: %s)" % \
                              (cover, e.message))
            else:
                self.w_cover.signal_emit("cover,show", "")
                return
        self.w_cover.signal_emit("cover,hide", "")
        self.cover.hide()

    def _setup_playlist_status(self, repeat=None, shuffle=None):
        self.repeat = repeat
        if repeat:
            self.pl.signal_emit("repeat,yes", "")
        else:
            self.pl.signal_emit("repeat,no", "")

        self.shuffle = shuffle
        if shuffle:
            self.pl.signal_emit("shuffle,yes", "")
        else:
            self.pl.signal_emit("shuffle,no", "")

    def _setup_rating(self):
        if self.model.rating is None:
            self.media_info.signal_emit("rating,hide", "")
            return False

        self.media_info.signal_emit("rating,show", "")
        try:
            rating = int(self.model.rating) / 5.0
        except ValueError:
            rating = 0.0
        self.rating_bar.part_drag_value_set("knob", rating, 0.0)
        return True


    @edje.decorators.signal_callback("ratingbar_base,drag_set",
                                     "player_audio/media_info:"
                                     "rating_area:*")
    def rating_bar_cb(obj, emission, source):
        if obj.callback_set_rating:
            pos = float(source.split(":")[-2].replace(",", "."))
            rating = int(round(round(pos, 1) * 5))
            obj.callback_set_rating(rating)

    def theme_changed(self):
        EdjeWidget.theme_changed(self)
        self.setup_widgets()
        self.setup_model(self.model)

    def delete(self):
        self.w_cover.part_unswallow(self.cover)
        self.cover.delete()
        self.signal_callback_del("mouse,down,1",
                                 "player_audio/media_info:track_title",
                                 self.scroll_start)
        self.media_info.part_unswallow(self.title)
        self.title.stop()
        self.title.delete()
        self.model = None
        EdjeWidget.delete(self)


class AudioPlayerController(PlayerController):
    terra_type = "Controller/Media/Audio"
    player_type = "Player/Audio"
    hooks_type = "Hook/Player/Audio"
    name = "audio"

    def __init__(self, model, canvas, parent):
        PlayerController.__init__(self, model, canvas, parent)
        self.view.callback_theme_changed = self.theme_changed
        self.log = logging.getLogger("canola.plugins.audio.player")
        self.repeat_callback = self.cb_repeat
        self.shuffle_callback = self.cb_shuffle
        self.setup_remote()
        self.audio_screen = AudioPlayerScreen(self.model, self.repeat,
                                              self.shuffle, self.view)
        self.audio_screen.callback_set_rating = self.set_rating
        self.view.throbber_start()
        self.view.signal_callback_add("transition,in,finished", "*",
                                      self.transition_in_finished_cb)
        self.view.signal_callback_add("transition,in,finished", "*",
                                      self.audio_screen.scroll_start)
        self.view.signal_callback_add("transition,out,finished", "*",
                                      self.audio_screen.scroll_stop)
        self.setup_view()

    def transition_in_finished_cb(self, obj, emission, source):
        self.view.evas.render() # XXX: load_atabake() may take too long
                                # XXX: force redraw to clear last frame
                                # XXX: and avoid idler_add()
        self.load_atabake()
        self.setup_interface()
        self.setup_model()
        self.view.signal_callback_del("transition,in,finished", "*",
                                      self.transition_in_finished_cb)

    def do_resume(self):
        self.audio_screen.cover.reload()
        self.view.setup_volume(self._volume)

    def cb_repeat(self):
        self.audio_screen._setup_playlist_status(self.repeat, self.shuffle)

    def cb_shuffle(self):
        self.audio_screen._setup_playlist_status(self.repeat, self.shuffle)

    def set_rating(self, rating):
        if self.model.rating is not None:
            self.model.rating = rating
            self.commit_changes()

    def setup_model(self, view=True):
        self.setup_view()
        self.view.setup_volume(self._volume)
        self.current = self.model.parent.current
        self.set_uri(self.model.uri)
        self.play()

    def setup_view(self):
        self.view.setup_view()
        self.audio_screen.setup_model(self.model)
        self.view.part_swallow("player_view/media_area", self.audio_screen)
        self._remote.model = self.model

    def setup_remote(self):
        self._remote = RemotePlayer()
        self._remote.model = self.model
        self._remote.state = self.state
        self._remote.callback_prev = self.prev
        self._remote.callback_next = self.next
        self._remote.callback_get_volume = self.get_volume
        self._remote.callback_set_volume = self.set_volume
        self._remote.callback_handle_play_pause = self.handle_play_pause

    def next_state(self, enabled):
        if enabled:
            self.view.media_controls.signal_emit("forward,enable", "")
            #self._remote.view.signal_emit("forward,enable", "")
        else:
            self.view.media_controls.signal_emit("forward,disable", "")
            #self._remote.view.signal_emit("forward,disable", "")

    def previous_state(self, enabled):
        if enabled:
            self.view.media_controls.signal_emit("previous,enable", "")
            #self._remote.view.signal_emit("previous,enable", "")
        else:
            self.view.media_controls.signal_emit("previous,disable", "")
            #self._remote.view.signal_emit("previous,enable", "")

    def _player_eos(self):
        try:
            count = int(self.model.playcnt)
        except ValueError:
            count = 0

        self.model.playcnt = count + 1
        self.commit_changes()
        PlayerController._player_eos(self)

    def _player_state_changed(self, state):
        if state == self.STATE_PLAYING:
            self.audio_screen.scroll_start()
        PlayerController._player_state_changed(self, state)
        self._remote.state = self.state

    def _player_buffering(self, value):
        PlayerController._player_buffering(self, value)
        if value == 100.0 and self.model.start_pos:
            self.seek(self.model.start_pos)
            self.model.start_pos = None

    def theme_changed(self):
        self.audio_screen.theme_changed()
        self.setup_view()
        if self.state != self.STATE_PLAYING:
            self.view.media_controls.signal_emit("media,state,pause", "")
        self.update_trackbar()

    def back(self):
        self.conf.save()
        self.parent.back()

    def go_home(self):
        self.conf.save()
        self.parent.go_home()

    def must_hold(self):
        return True

    def delete(self):
        self.audio_screen.delete()
        self._remote.delete()
        self._remote = None
        PlayerController.delete(self)
        self.model = None

    def options_model_get(self):
        return AudioPlayerOptionsModel(None, self)


GenericListController = mger.get_class("Controller/Folder")
AudioOnTheGoModelFolder = mger.get_class("Model/Folder/Media/Audio/" \
                                         "Local/Playlist/OnTheGo")
class AudioListController(GenericListController):
    terra_type = "Controller/Folder/Media/Audio/Local"

    message = "Adding folder to \"On the Move\" playlist..."
    step = 10
    status_timer = None
    status_timeout = 0.2
    min_dialog_delay = 1.0

    def __init__(self, model, canvas, parent):
        GenericListController.__init__(self, model, canvas, parent)
        if self.model.has_onthego:
            self._dialog_hide_timer = None
            self._folderotg_idler = None
            self._onthego = AudioOnTheGoModelFolder()
            self.view._list.hold_cb = self._onthego_cb
            self._dialog = ThrobberDialog(self.message, self._cancel_clicked)
        else:
            self._onthego = None
            self.view._list.hold_cb = None

    def _cancel_clicked(self, ignored, action):
        if self.status_timer is not None:
            self.status_timer.stop()
            self.status_timer = None

        if action:
            if self._dialog_hide_timer:
                self._dialog_hide_timer.stop()
                self._dialog_hide_timer = None

            assert self._folderotg_idler != None
            self._folderotg_idler.stop()
            self._folderotg_idler = None
            self._songs_added_msg(morethan1=True)

    def _songs_added_msg(self, morethan1=False):
        if morethan1:
            self.view.bottom_message_show(
                "%d songs added to playlist" % self.processed_files, 0.8)
        else:
            self.view.bottom_message_show("Song added to playlist", 0.8)

    def _onthego_cb(self, idx):
        model = self.model.children[idx]
        if isinstance(model, ModelFolder):
            self.parent.show_notify(self._dialog)
            self.dialog_creation = ecore.loop_time_get()
            self.processed_files = 0
            self._status_timer = ecore.timer_add(self.status_timeout,
                                                 self._status_report)
            self._folderotg_idler = ecore.idler_add(
                self._folder_onthego_add,
                [0, model, [], self._onthego_finish_cb])
        else:
            self._songs_added_msg()
            self._onthego.enqueue(model)
        return True

    def _status_report(self):
        self._dialog.message_set(
            "  %d files processed..." % self.processed_files)
        return True

    def _onthego_finish_cb(self):
        if self.status_timer is not None:
            self.status_timer.stop()
            self.status_timer = None

        d = ecore.loop_time_get() - self.dialog_creation
        d = self.min_dialog_delay - d
        if d > 0:
            def hide_dialog():
                self._dialog_hide_timer = None
                self._dialog.hide()
                self._songs_added_msg(morethan1=True)
                return False

            self._dialog_hide_timer = ecore.timer_add(d, hide_dialog)
        else:
            self._dialog.hide()
            self._songs_added_msg(morethan1=True)

    def _folder_onthego_add(self, ctx):
        start, model, dirs, end_cb = ctx

        end = start + self.step
        if start == 0:
            model.load()

        for child in model.children[start:end]:
            if hasattr(child, "only_children_otg"):
                continue
            elif isinstance(child, ModelFolder):
                dirs.append(child)
            else:
                self._onthego.enqueue(child)
                self.processed_files += 1

        if end < len(model.children):
            ctx[0] = end
            return True

        model.unload()

        if not dirs:
            if end_cb:
                end_cb()
            return False

        def process_dir(d, dirs, end_cb):
            def end_cb_aux():
                if not dirs:
                    if end_cb:
                        end_cb()
                else:
                    d = dirs.pop(0)
                    process_dir(d, dirs, end_cb)

            self._folderotg_idler = ecore.idler_add(
                self._folder_onthego_add, [0, d, [], end_cb_aux])

        d = dirs.pop(0)
        process_dir(d, dirs, end_cb)

        return False


PlayerCurrentMediaURI = mger.get_class("Current/MediaURI")
def row_value_set(self, v):
    if PlayerCurrentMediaURI.uri == v.uri and PlayerCurrentMediaURI.model is v:
        self.signal_emit("show,marker", "")
    else:
        self.signal_emit("hide,marker", "")

    self.part_text_set("text", v.name)


ResizableRowRendererWidget = mger.get_class("Widget/ResizableRowRenderer")
class ResizableRowRendererCurrentURIWidget(ResizableRowRendererWidget):
    terra_type = "Widget/ResizableRowRendererURI"

    row_group = "list_current_item_text_resizable"
    do_value_set = row_value_set

    def __init__(self, parent, theme=None):
        ResizableRowRendererWidget.__init__(self, parent, theme)
        PlayerCurrentMediaURI.callbacks.append(self.player_uri_changed)
        self.event_callback_add(evas.EVAS_CALLBACK_DEL,
                                self.player_uri_changed_uregister)

    def player_uri_changed_uregister(self, o):
        PlayerCurrentMediaURI.callbacks.remove(self.player_uri_changed)

    def player_uri_changed(self, PlayerCurrentMediaURI):
        if self._model is not None and \
                PlayerCurrentMediaURI.uri == self._model.uri and \
                PlayerCurrentMediaURI.model is self._model:
            self.signal_emit("show,marker", "")
        else:
            self.signal_emit("hide,marker", "")


RowRendererWidget = mger.get_class("Widget/RowRenderer")
class RowRendererCurrentURIWidget(RowRendererWidget):
    terra_type = "Widget/RowRendererURI"

    row_group = "list_current_item_text"
    do_value_set = row_value_set
    resizable_row_renderer = ResizableRowRendererCurrentURIWidget

    def __init__(self, parent, theme=None):
        RowRendererWidget.__init__(self, parent, theme)
        PlayerCurrentMediaURI.callbacks.append(self.player_uri_changed)
        self.event_callback_add(evas.EVAS_CALLBACK_DEL,
                                self.player_uri_changed_uregister)

    def player_uri_changed_uregister(self, o):
        PlayerCurrentMediaURI.callbacks.remove(self.player_uri_changed)

    def player_uri_changed(self, PlayerCurrentMediaURI):
        if self._model is not None and \
                PlayerCurrentMediaURI.uri == self._model.uri and \
                PlayerCurrentMediaURI.model is self._model:
            self.signal_emit("show,marker", "")
        else:
            self.signal_emit("hide,marker", "")


class BaseDeletableRowRendererURI(BaseDeletableRowRenderer):
    def value_set(self, model):
        if not model or model is self._model:
            return

        if model.parent.is_removable_child(model):
            self.signal_emit("delete_button,show", "")
        else:
            self.signal_emit("delete_button,hide", "")

        # XXX: faster hash lookups
        if not hasattr(model, "_deletable_select_state"):
            model._deletable_select_state = False

        if model._deletable_select_state:
            self.delete_button.state_set(ActionButton.STATE_DELETE)
        else:
            self.delete_button.state_set(ActionButton.STATE_TRASH)

        self._model = model
        self.part_text_set("text", model.name)

        if PlayerCurrentMediaURI.uri == model.uri and \
                PlayerCurrentMediaURI.model is model:
            self.signal_emit("show,marker", "")
        else:
            self.signal_emit("hide,marker", "")


class ResizableDeletableRowRendererURIWidget(BaseDeletableRowRendererURI,
                                             ResizableRowRendererCurrentURIWidget):
    row_group = "list_current_item_text_resizable"

    def __init__(self, parent, theme=None):
        BaseDeletableRowRendererURI.__init__(self, parent, theme)
        PlayerCurrentMediaURI.callbacks.append(self.player_uri_changed)
        self.event_callback_add(evas.EVAS_CALLBACK_DEL,
                                self.player_uri_changed_uregister)


class DeletableRowRendererURIWidget(BaseDeletableRowRendererURI):
    row_group = "list_current_item_text_deletable"
    resizable_row_renderer = ResizableDeletableRowRendererURIWidget

    def __init__(self, parent, theme=None):
        BaseDeletableRowRendererURI.__init__(self, parent, theme)
        PlayerCurrentMediaURI.callbacks.append(self.player_uri_changed)
        self.event_callback_add(evas.EVAS_CALLBACK_DEL,
                                self.player_uri_changed_uregister)

    def player_uri_changed_uregister(self, o):
        PlayerCurrentMediaURI.callbacks.remove(self.player_uri_changed)

    def player_uri_changed(self, PlayerCurrentMediaURI):
        if self._model is not None and \
                PlayerCurrentMediaURI.uri == self._model.uri and \
                PlayerCurrentMediaURI.model is self._model:
            self.signal_emit("show,marker", "")
        else:
            self.signal_emit("hide,marker", "")


def directory_row_value_set(self, value):
    self.part_text_set("text", value.name)
    if isinstance(value, DirectoryModel):
        icon_signal = "icon,show"
        marker_signal = "hide,marker"
    else:
        icon_signal = "icon,hide"
        if PlayerCurrentMediaURI.uri == value.uri and \
                PlayerCurrentMediaURI.model is value:
            marker_signal = "show,marker"
        else:
            marker_signal = "hide,marker"
    self.signal_emit(icon_signal, "")
    self.signal_emit(marker_signal, "")

class DirectoryResizableRowRendererCurrentURIWidget(ResizableRowRendererCurrentURIWidget):
    row_group = "list_item_fs_resizable"
    do_value_set = directory_row_value_set

    def player_uri_changed(self, PlayerCurrentMediaURI):
        if self._model is not None \
                and not isinstance(self._model, DirectoryModel) \
                and PlayerCurrentMediaURI.uri == self._model.uri \
                and PlayerCurrentMediaURI.model is self._model:
            self.signal_emit("show,marker", "")
        else:
            self.signal_emit("hide,marker", "")


class DirectoryRowRendererCurrentURIWidget(RowRendererCurrentURIWidget):
    row_group = "list_item_fs"
    do_value_set = directory_row_value_set
    resizable_row_renderer = DirectoryResizableRowRendererCurrentURIWidget

    def player_uri_changed(self, PlayerCurrentMediaURI):
        if self._model is not None \
                and not isinstance(self._model, DirectoryModel) \
                and PlayerCurrentMediaURI.uri == self._model.uri \
                and PlayerCurrentMediaURI.model is self._model:
            self.signal_emit("show,marker", "")
        else:
            self.signal_emit("hide,marker", "")



class DirectoryAudioListController(AudioListController):
    terra_type = "Controller/Folder/Directory/Media/Audio"
    row_renderer = DirectoryRowRendererCurrentURIWidget
    list_group = "list"


class AudioLocalPlaylistController(AudioListController):
    terra_type = "Controller/Folder/Media/Audio/Local/Playlists"
    row_renderer = DeletableRowRendererWidget

    def __init__(self, model, canvas, parent):
        AudioListController.__init__(self, model, canvas, parent)
        self._otg_hook = AudioPlaylistOTGHook(self)
        self.parent.otg_hooks.add(self._otg_hook)
        self.model.add_child_removed_callback(self._song_removed)

    def delete(self):
        self.model.del_child_removed_callback(self._song_removed)
        self.parent.otg_hooks.remove(self._otg_hook)
        self._otg_hook = None
        AudioListController.delete(self)

    def _song_removed(self, model):
        curr_song = self.parent.get_playing_model()
        if curr_song and curr_song.parent.__class__ == model.__class__ and \
           curr_song.parent.id == model.id:
            self.parent.kill_player()


OnTheGoListScreen = mger.get_class("Widget/ListScreen/OnTheGo")
OptionsControllerMixin = mger.get_class("OptionsControllerMixin")
class OTGAudioListController(GenericListController, OptionsControllerMixin):
    terra_type = "Controller/Folder/Media/Audio/Local/Playlist/OnTheGo"
    view_class = OnTheGoListScreen
    row_renderer = DeletableRowRendererURIWidget
    PlaylistClass = mger.get_class("Model/Folder/Media/Audio/Local/Playlist/Songs")

    def __init__(self, model, canvas, parent):
        GenericListController.__init__(self, model, canvas, parent)
        OptionsControllerMixin.__init__(self)
        self.model.add_child_removed_callback(self._song_removed)
        self._hooks = parent.otg_hooks

    def delete(self):
        self._hooks = None
        self.model.del_child_removed_callback(self._song_removed)
        GenericListController.delete(self)
        OptionsControllerMixin.delete(self)

    def _song_removed(self, model):
        curr_song = self.parent.get_playing_model()
        # If listening to OTG and delete curr song being played, stop player.
        # Must use "is" instead of "==" or else would stop if playing back the
        # same song but from other ModelFolder than OTG.
        if model is curr_song:
            self.parent.kill_player()
        if not model.parent.children:
            # XXX - Find a better way to remove options button
            # Accessing TaskController.MainWindow
            self.parent.view._recheck_options_visibility(transition=True)

    def clear_playlist(self):
        self.model.clear_playlist()
        self._hooks.inform_playlist_cleared()
        # XXX - Find a better way to remove options button
        # Acessing TaskController.MainWindow
        self.parent.view._recheck_options_visibility()
        curr_song = self.parent.get_playing_model()
        if curr_song and isinstance(curr_song.parent, AudioOnTheGoModelFolder):
            self.parent.kill_player()

    def save_playlist(self, text, force=False):
        return self.model.save_playlist(text, force=force)

    def playlist_saved(self, playlist_id):
        """Will be responsible for updating the Playlist Model Folder if
        alive or kill the player if not possible.
        """
        self._hooks.inform_playlist_saved(playlist_id)

    def has_options(self):
        if self.model.children:
            return OptionsControllerMixin.has_options(self)
        else:
            return False

    def options_model_get(self):
        return AudioOnTheGoOptionsMF(None, self)


ModalController = mger.get_class("Controller/Modal")
MessageModal = mger.get_class("Widget/Settings/MessageModal")
class OTGOptionsSaveController(ModalController):
    terra_type = "Controller/Options/Action/Media/Audio/Local/Playlist/OnTheGo/Save"

    def __init__(self, model, canvas, parent):
        ModalController.__init__(self, model, canvas, parent)
        self.model = model
        self.view = None
        self._id = None
        self._text = ""
        self._setup = {"ChooseName": self._choose_name_setup,
                       "Replace": self._replace_setup,
                       "Done": self._done_setup}

        self._state = "ChooseName"
        self._update_state()

    def _update_state(self):
        if self.view:
            self.view.delete()
            self.view = None

        self._setup[self._state]()

    def _choose_name_setup(self):
        self.view = EntryModal(self.parent.last_panel, self.model.name,
                               "Choose playlist name:", value=self._text)
        self.view.callback_escape = self.close
        self.view.callback_cancel_clicked = self.close

        def ok_cb(text):
            text = os.path.basename(text) or "NewPlaylist"
            saved = self.model.save_playlist(text)
            if saved:
                self._id, self._text = saved
                self._text = os.path.basename(self._text[:-4])
                self._state = "Done"
                self.update_view()
            else:
                self._text = text
                self._state = "Replace"
                self.update_view()

        self.view.callback_ok_clicked = ok_cb
        self.view.show()

    def _replace_setup(self):
        msg = "There is already a playlist with this name. Do you want " \
              "to replace it?"
        self.view = MessageModal(self.parent.last_panel, self.model.name, msg,
                                 label_ok="   Yes   ",
                                 label_cancel="    No    ")

        def yes_cb():
            saved = self.model.save_playlist(self._text, force=True)
            self._id, self._text = saved
            self._text = os.path.basename(self._text[:-4])
            self._state = "Done"
            self.update_view()

        def no_cb():
            self._state = "ChooseName"
            self.update_view()

        self.view.callback_escape = no_cb
        self.view.callback_cancel_clicked = no_cb
        self.view.callback_ok_clicked = yes_cb
        self.view.show()

    def _done_setup(self):
        msg = "The playlist '%s' was saved." % self._text
        self.view = MessageModal(self.parent.last_panel, self.model.name, msg,
                                 has_cancel=False)

        self.view.callback_escape = self.close
        self.view.callback_ok_clicked = self.close
        self.parent.screen_controller.playlist_saved(self._id)
        self.view.show()

    def update_view(self):
        def cb(*ignored):
            self._update_state()
        self.view.hide(end_callback=cb)

    def close(self):
        def cb(*ignored):
            self.back()
            self.parent.back()
        self.view.hide(end_callback=cb)

    def delete(self):
        self.view.delete()
        self.view = None
        self.model = None


class OTGOptionsClearController(ModalController):
    terra_type = "Controller/Options/Action/Media/Audio/Local/Playlist/OnTheGo/Clear"

    def __init__(self, model, canvas, parent):
        ModalController.__init__(self, model, canvas, parent)
        self.model = model
        msg = "Are you sure you want to clear the playlist?"
        self.view = MessageModal(self.parent.last_panel, self.model.name, msg,
                                 label_ok="   Yes   ",
                                 label_cancel="    No    ")

        def yes_cb():
            self.model.clear_playlist()
            self.close()

        self.view.callback_escape = self.close
        self.view.callback_cancel_clicked = self.close
        self.view.callback_ok_clicked = yes_cb
        self.view.show()

    def close(self):
        def cb(*ignored):
            self.back()
            self.parent.back()
        self.view.hide(end_callback=cb)

    def delete(self):
        self.view.delete()
        self.view = None
        self.model = None


class AudioLocalAllControllerFolder(AudioListController):
    terra_type = "Controller/Folder/Media/Audio/Local/All"
    row_renderer = RowRendererCurrentURIWidget


class AudioLocalRandomControllerFolder(AudioListController):
    terra_type = "Controller/Folder/Media/Audio/Local/Random"
    row_renderer = RowRendererCurrentURIWidget


class AudioLocalPlaylistMediaControllerFolder(AudioListController):
    terra_type = "Controller/Folder/Media/Audio/Local/Playlist/Songs"
    row_renderer = RowRendererCurrentURIWidget


class AudioLocalArtistMediaControllerFolder(AudioListController):
    terra_type = "Controller/Folder/Media/Audio/Local/Artist/Songs"
    row_renderer = RowRendererCurrentURIWidget


class AudioLocalAlbumMediaControllerFolder(AudioListController):
    terra_type = "Controller/Folder/Media/Audio/Local/Album/Songs"
    row_renderer = RowRendererCurrentURIWidget


class AudioLocalRatingControllerFolder(AudioListController):
    terra_type = "Controller/Folder/Media/Audio/Local/Rating"
    row_renderer = RowRendererCurrentURIWidget


class AudioLocalFavouritesControllerFolder(AudioListController):
    terra_type = "Controller/Folder/Media/Audio/Local/Favourites"
    row_renderer = RowRendererCurrentURIWidget


class AudioLocalGenreMediaControllerFolder(AudioListController):
    terra_type = "Controller/Folder/Media/Audio/Local/Genre/Songs"
    row_renderer = RowRendererCurrentURIWidget
