# Canola2 Youtube Plugin
# Copyright (C) 2008 Instituto Nokia de Tecnologia
# Author: Adriano Rezende <adriano.rezende@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import os
import glob
import time
import evas
import ecore
import ecore.evas

from terra.core.manager import Manager
from terra.core.plugin_prefs import PluginPrefs
from terra.core.threaded_func import ThreadedFunction


manager = Manager()
SystemProps = manager.get_class("SystemProperties")
system_props = SystemProps()

THUMB_CACHE_SIZE = 500 # keep 500 thumbs


def get_thumb_path(id=None):
    prefs = PluginPrefs("settings")
    try:
        path = prefs["youtube_thumbs"]
    except KeyError:
        path = os.path.join(os.path.expanduser("~"),
                            ".canola", "youtube", "thumbs")
        prefs["youtube_thumbs"] = path
        prefs.save()

    if not os.path.exists(path):
        os.makedirs(path)

    if id is not None:
        path = os.path.join(path, "%s.jpg" % id)

    return path


def space_available(path, bytes_required):
    try:
        bytes_avail = system_props.bytes_available(path)
        return (bytes_avail >= bytes_required)
    except OSError:
        return False


def get_video_path(id=None):
    prefs = PluginPrefs("settings")
    try:
        path = prefs["download_path"]
        path = os.path.join(path, "youtube")
    except KeyError:
        path = os.path.join(os.path.expanduser("~"),
                            ".canola", "youtube", "videos")

    if id is not None:
        path = os.path.join(path, "%s.flv" % id)

    return path


def round_rating(value):
    """Round the rating value to fit on the following pre-defined values:
    (0.0, 0.5, 1.0, 1.5, 2.0, ..., 5.0), and return the percentage value
    of this rounded rating, dividing by 5.0."""
    if value is None:
        value = 0
    rounded_rating = round(value, 1)
    if 3 <= int((rounded_rating % 1) * 10) <= 7:
        rounded_rating = int(rounded_rating) + 0.5
    else:
        rounded_rating = round(rounded_rating)
    return float(rounded_rating) / 5.0


def remove_old_thumbs(thumb_path, video_path):
    """Remove old thumbnails based on last access time."""
    lst = []
    nw = time.time()
    for filepath in glob.glob(os.path.join(thumb_path, "*.jpg")):
        path, name = os.path.split(filepath)
        vfilepath = os.path.join(video_path, name[:-4] + ".flv")

        # do not remove local videos thumbnails
        if os.path.exists(vfilepath):
            continue

        st = os.stat(filepath)
        lst.append((filepath, nw - st.st_atime))

    lst.sort(lambda a, b: int(a[1]) - int(b[1]))

    for filepath, lifetime in lst[THUMB_CACHE_SIZE:]:
        os.unlink(filepath)


# Remove old thumbnails on module loading
thumb_path = get_thumb_path()
video_path = get_video_path()
ThreadedFunction(None, remove_old_thumbs, thumb_path, video_path).start()
