# Canola2 Youtube Plugin
# Copyright (C) 2008 Instituto Nokia de Tecnologia
# Author: Adriano Rezende <adriano.rezende@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import evas
import ecore
import locale
import logging

from terra.core.manager import Manager
from terra.ui.base import PluginThemeMixin

from model import SearchModelFolder, PlayerOptionsModel, \
    HistoryModelFolder, HistoryOptionsModel, MyVideosModelFolder


manager = Manager()
GeneralActionButton = manager.get_class("Widget/ActionButton")
EntryDialogModel = manager.get_class("Model/EntryDialog")
BaseListController = manager.get_class("Controller/Folder")
BaseVideoPlayerController = manager.get_class("Controller/Media/Video")
BaseRowRenderer =  manager.get_class("Widget/RowRenderer")
ResizableRowRenderer = manager.get_class("Widget/ResizableRowRenderer")
OptionsControllerMixin = manager.get_class("OptionsControllerMixin")

locale.setlocale(locale.LC_ALL, "")
log = logging.getLogger("plugins.canola-tube.ui")


class ListController(BaseListController):
    """YouTube Navigation List.

    This is the initial list of YouTube plugin. It shows all navigation
    possibilities, like Search for videos, Bookmarks, History, Most Viewed,
    Most Recent, Top Rated, ...

    This list is identical to Canola Text List, with a special treatment for
    the Search item, that popup a Search Modal before go to the next screen.

    @see: BaseListController
    """
    terra_type = "Controller/Folder/Task/Video/YouTube"

    def cb_on_clicked(self, view, index):
        model = self.model.children[index]

        if type(model) is not SearchModelFolder:
            BaseListController.cb_on_clicked(self, view, index)
            return

        def do_search(ignored, text):
            if text is not None:
                model.query = text
                BaseListController.cb_on_clicked(self, view, index)

        dialog = EntryDialogModel("Search for videos", "Enter keywords or tags:",
                                  answer_callback=do_search)
        self.parent.show_notify(dialog)


class ActionButton(PluginThemeMixin, GeneralActionButton):
    plugin = "youtube"


class GeneralRowRenderer(PluginThemeMixin, BaseRowRenderer):
    """YouTube Base List Item Renderer.

    This renderer is applied on ServiceController. Providing
    a personalized list item for videos. It shows the following video
    properties: title, thumbnail image, rating, view count and author.

    @note: This renderer extends BaseRowRenderer, overloading
    some methods.

    @note: An instance of this class can be reused to show properties
    of diferent models due to optimizations. So, to take full advantages
    of this feature, you can load heavy data like images, on value_set method,
    only keeping in the models the path to load the data. This has a nice
    effect with memory usage and is very scalable. For example if you have a
    list with 6 renderers, and your list has 800 images, only 6 images are
    stored in memory, not 800 (generally the number of renderers is close to
    the number of visible items on the list).

    @note: To tell Canola to look first on the plugin theme you have to
    add PluginThemeMixin class and set the plugin variable with the plugin
    name.

    @see: ServiceController, BaseRowRenderer, PluginThemeMixin
    """
    plugin = "youtube"

    def __init__(self, parent, theme=None):
        BaseRowRenderer.__init__(self, parent, theme)
        self.image = self.evas.FilledImage()
        self.part_swallow("contents", self.image)
        self.signal_emit("thumb,hide", "")

        self.bg_selection = self.PluginEdjeWidget("widget/list/bg_selection")
        self.part_swallow("selection", self.bg_selection)

        self.rating_area = self.PluginEdjeWidget("widget/rating_bar_small")
        self.part_swallow("rating_area", self.rating_area)

        self.delete_button = ActionButton(self)
        self.delete_button.state_set(ActionButton.STATE_TRASH)
        self.delete_button.on_button_delete_pressed_set(self.cb_delete_pressed)
        self.part_swallow("delete_button", self.delete_button)

        self.delete_button.on_contents_box_expanded_set(self.cb_box_expanded)
        self.delete_button.on_contents_box_collapsed_set(self.cb_box_collapsed)
        self.delete_button.disable_download()

    def theme_changed(self, end_callback=None):
        def cb(*ignored):
            self.part_swallow("selection", self.bg_selection)
            self.part_swallow("rating_area", self.rating_area)
            self.part_swallow("delete_button", self.delete_button)
            self.part_swallow("contents", self.image)
            if end_callback is not None:
                end_callback(self)

        self.bg_selection.theme_changed()
        self.rating_area.theme_changed()
        self.delete_button.theme_changed()
        self.delete_button.state_set(ActionButton.STATE_TRASH)
        self.delete_button.disable_download()

        BaseRowRenderer.theme_changed(self, cb)

    def cb_box_expanded(self, *ignored):
        self._model.selected_state = True

    def cb_box_collapsed(self, *ignored):
        self._model.selected_state = False

    def cb_delete_pressed(self, *ignored):
        def cb_collapsed(*ignored):
            self.delete_button.signal_callback_del("contents_box,collapsed", "",
                                                   cb_collapsed)
            self._model.parent.delete_model(self._model.id)
            self._model.parent.children.remove(self._model)
            self.delete_button.signal_emit("unblock,events", "")
            self.delete_button.state_set(ActionButton.STATE_TRASH)

        self.delete_button.signal_callback_add("contents_box,collapsed", "",
                                               cb_collapsed)

    def cb_load_thumbnail(self):
        try:
            self.image.file_set(self._model.thumb)
            self.signal_emit("thumb,show", "")
        except Exception, e:
            log.error("could not load image %r: %s", self._model.thumb, e)
            self.signal_emit("thumb,hide", "")

    def value_set(self, model):
        """Apply the model properties to the renderer."""
        if not model or model is self._model:
            return

        if model.parent.delete_possible:
            self.signal_emit("delete_button,show", "")
        else:
            self.signal_emit("delete_button,hide", "")

        if not hasattr(model, "selected_state"):
            model.selected_state = False

        if model.selected_state:
            self.delete_button.state_set(ActionButton.STATE_DELETE)
        else:
            self.delete_button.state_set(ActionButton.STATE_TRASH)

        if isinstance(model.parent, MyVideosModelFolder):
            self.delete_button.delete_text_set("Delete")
        else:
            self.delete_button.delete_text_set("Remove")

        self._model = model
        self.part_text_set("text", model.name)
        self.part_text_set("duration", model.duration_formated)
        self.part_text_set("from", "From: " + model.artist)
        self.part_text_set("views", locale.format("%d", model.view_count, True) \
                               + " views")
        self.rating_area.part_drag_value_set("knob", model.rounded_rating, 0.0)

        model.request_thumbnail(self.cb_load_thumbnail)

    @evas.decorators.del_callback
    def __on_delete(self):
        """Free internal data on delete."""
        self.image.delete()
        self.rating_area.delete()
        self.bg_selection.delete()
        self.delete_button.delete()


class ResizableRowRendererWidget(GeneralRowRenderer, ResizableRowRenderer):
    """YouTube Base List Item Renderer for Selected Items.

    This renderer is very similar with RowRendererWidget. The diference
    is the select animation that it starts.

    @see: ServiceController, RowRendererWidget
    """
    row_group="list_item_youtube_resizeable"

    def __init__(self, parent, theme=None):
        GeneralRowRenderer.__init__(self, parent, theme)


class RowRendererWidget(GeneralRowRenderer):
    row_group="list_item_youtube"
    resizable_row_renderer = ResizableRowRendererWidget


class ServiceController(BaseListController, OptionsControllerMixin):
    """YouTube Video List.

    This list is like a page result that shows the videos that match
    with some criteria.

    @note: This class extends BaseListController, but apply a different
    item renderer declaring a row_renderer variable and a different
    screen interface declaring a list_group variable. The group "list_video"
    is the interface of the default Canola video list.

    @see: BaseListController, RowRendererWidget,
          SelectedRowRendererWidget
    """
    terra_type = "Controller/Folder/Task/Video/YouTube/Service"
    row_renderer = RowRendererWidget
    list_group = "list_video"

    def __init__(self, model, canvas, parent):
        self.empty_msg = model.empty_msg
        BaseListController.__init__(self, model, canvas, parent)
        OptionsControllerMixin.__init__(self)
        self.model.callback_notify = self._show_notify

    def _show_notify(self, err):
        """Popup a modal with a notify message."""
        self.parent.show_notify(err)

    def options_model_get(self):
        if isinstance(self.model, HistoryModelFolder):
            return HistoryOptionsModel(None, self)


###########################################################################
# Video Player
###########################################################################

class VideoPlayerController(BaseVideoPlayerController):
    """YouTube Video Player Controller.

    This class extends the Canola Video Player.
    """
    terra_type = "Controller/Media/Video/YouTubeLocal"

    # disable unused properties from parent
    repeat = None
    shuffle = None
    saved_pos = 0

    INITIAL_VIDEO_SIZE = (1500 * 1024)

    def __init__(self, model, canvas, parent):
        BaseVideoPlayerController.__init__(self, model, canvas, parent)
        self.started = False
        self.block_controls()
        self.disable_trackbar()
        self.view.set_tracking_state(enable=False)
        self.view.change_back_trackbar_state(enable=True)

    def show_notify(self, error):
        self.pause()
        self._player_state_changed(self.STATE_ERROR)
        self.video_screen.video.hide()
        self.parent.show_notify(error)

    def cb_download_progress(self, progress):
        self.update_back_trackbar(progress)

        if (self.model.bytes_written < self.INITIAL_VIDEO_SIZE \
                and self.model.progress < 1.0) or self.started:
            return

        self.setup_model(view=False, ready=True)

    def cb_download_finished(self, exception):
        self.update_back_trackbar(self.model.progress)
        if self.model.progress >= 1.0:
            self.model.save_milestone()

    def transition_in_finished_cb(self, obj, emission, source):
        BaseVideoPlayerController.transition_in_finished_cb(self, obj,
                                                            emission, source)
        self.start_video()

    def start_video(self):
        self.model.callback_notify = self.show_notify
        self.model.callback_download_progress = self.cb_download_progress
        self.model.callback_download_finished = self.cb_download_finished

        self.disable_trackbar()
        self.update_back_trackbar(0)

        self.started = False
        self.block_controls()

        self.model.do_load()

        if self.model.play_local:
            log.debug("playing video local")
            self.update_back_trackbar(1.0)
            self.setup_model(view=False, ready=True)

        self.view.set_playing(True)

    def play(self):
        if self.started:
            BaseVideoPlayerController.play(self)

    def pause(self):
        if self.started:
            BaseVideoPlayerController.pause(self)

    def prev(self):
        if not self.started or self.video_screen.fullscreen:
            return
        self.model.do_unload()
        BaseVideoPlayerController.prev(self)

    def next(self):
        if not self.started or self.video_screen.fullscreen:
            return
        self.model.do_unload()
        BaseVideoPlayerController.next(self)

    def update_model(self, new_model):
        self.model.callback_notify = None
        self.model.callback_download_progress = None
        self.model.callback_download_finished = None
        BaseVideoPlayerController.update_model(self, new_model)
        self.start_video()

    def setup_model(self, view=True, ready=False):
        if ready: # don't load until get uri
            self.started = True
            BaseVideoPlayerController.setup_model(self, view)
            self.unblock_controls()

    def _player_eos(self):
        log.debug("player received eos")
        return True

    def options(self):
        if self.started:
            BaseVideoPlayerController.options(self)

    def options_model_get(self):
        return PlayerOptionsModel(None, self)

    def back(self):
        BaseVideoPlayerController.back(self)
        self.model.do_unload()

    def delete(self):
        BaseVideoPlayerController.delete(self)
        self.model.do_unload()
