# Canola2 Youtube Plugin
# Copyright (C) 2008 Instituto Nokia de Tecnologia
# Author: Adriano Rezende <adriano.rezende@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import evas
import locale
import logging

from terra.core.manager import Manager
from terra.ui.modal import ModalMessage
from terra.ui.base import PluginEdjeWidget
from terra.ui.base import PluginThemeMixin
from terra.core.controller import Controller


manager = Manager()
ModalController = manager.get_class("Controller/Modal")
BasicPanel = manager.get_class("Controller/BasicPanel")
BaseScrollableText = manager.get_class("Widget/ScrollableTextBlock")

# format numbers to local format
locale.setlocale(locale.LC_ALL, "")
log = logging.getLogger("plugins.canola-tube.youtube.options")


class ModalMessageController(ModalController):
    def __init__(self, model, canvas, parent, message):
        ModalController.__init__(self, model, canvas, parent)

        self.view = ModalMessage(parent.last_panel.view,
                                 message, hborder=60, vborder=120)
        self.callback_leave = None
        self.view.callback_escape = self.close
        self.view.callback_clicked = self.close
        self.view.show()

    def close(self):
        def cb(*ignored):
            self.back(self.callback_leave)
        self.view.hide(end_callback=cb)

    def delete(self):
        self.view.delete()


class ScrollableTextBlock(PluginThemeMixin, BaseScrollableText):
    plugin = "youtube"
    group = "textblock_description"


class InfoOptionController(BasicPanel):
    terra_type = "Controller/Options/Folder/Player/YouTube/Info"

    def __init__(self, model, canvas, parent, theme=None):
        BasicPanel.__init__(self, model, canvas, parent)

        self.thumbnail = evas.FilledImage(canvas)

        self._body = PluginEdjeWidget(self.view.evas, "panel_info_youtube",
                                      self.view, plugin="youtube")

        self.description = ScrollableTextBlock(self.view.evas, self.view)
        self._body.part_swallow("description", self.description)

        self.rating_area = PluginEdjeWidget(self.view.evas,
                                            "widget/rating_bar_small",
                                            self.view, plugin="youtube")
        self.rating_area.signal_emit("rating,alternate", "")
        self._body.part_swallow("rating_area", self.rating_area)

        self.inner_contents_set(self._body)
        self.setup_information()

    def setup_information(self):
        model = self.model.screen_controller.model
        self._body.part_text_set("title", model.title)
        self._body.part_text_set("author", "From: " + model.artist)
        self._body.part_swallow("contents", self.thumbnail)
        self.description.text_set(model.description)
        self.rating_area.part_drag_value_set("knob",
                                             model.rounded_rating, 0.0)
        try:
            self.thumbnail.file_set(model.thumb)
            self._body.signal_emit("thumb,show", "")
        except Exception, e:
            self._body.signal_emit("thumb,hide", "")

    def delete(self):
        self.rating_area.delete()
        self._body.delete()
        self.thumbnail.delete()
        self.description.delete()
        BasicPanel.delete(self)


class SaveOptionController(ModalMessageController):
    terra_type = "Controller/Options/Folder/Player/YouTube/Save"

    def __init__(self, model, canvas, parent):
        if model.has_record():
            msg = "This video is already saved"
        elif not model.downloaded():
            msg = "In order to save you must complete the download"
        else:
            if model.execute():
                msg = "This video has been saved to disk"
            else:
                msg = "Could not save this video to disk"
        ModalMessageController.__init__(self, model, canvas, parent, msg)

    def close(self):
        ModalMessageController.close(self)


class BookmarkOptionController(ModalMessageController):
    terra_type = "Controller/Options/Folder/Player/YouTube/Bookmark"

    def __init__(self, model, canvas, parent):
        if model.has_record():
            msg = "This video is already on bookmark"
        else:
            if model.execute():
                msg = "This video has been added to bookmark"
            else:
                msg = "Could not add this video to bookmark"
        ModalMessageController.__init__(self, model, canvas, parent, msg)

    def close(self):
        ModalMessageController.close(self)
