/*
 *  This file is part of CacheMe
 *
 *  (c) 2011 by Till Harbaum
 */

#include <QDebug>
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QGroupBox>
#include <QToolButton>
#include <QDialogButtonBox>
#include <QComboBox>
#include <math.h>

#include "coordinatetool.h"
#include "pickermanager.h"
#include "filefinder.h"
#include "swapbox.h"

// a hbox with two strings, one left aligned and one right aligned
class InfoWidget : public QWidget {
public:
  InfoWidget(const QString &leftStr = "",
	     const QString &rightStr = "",
	     QWidget *parent = 0) : QWidget(parent) {

    QHBoxLayout *layout = new QHBoxLayout;
    layout->setContentsMargins(0,0,0,0);
    layout->setSpacing(0);

    m_left = new QLabel(leftStr + ":");
    layout->addWidget(m_left);
    layout->addStretch();
    m_right = new QLabel("<b>" + rightStr + "</b>");
    layout->addWidget(m_right);

    setLayout(layout);
  }

  QLabel *getLabel() {
    return m_right;
  }

private:
  QLabel *m_left, *m_right;
};

void CoordinateTool::acceptCoordinate() {
  // fetch coordinate from dialog
  m_latManager->evaluate();
  m_lonManager->evaluate();

  // update labels
  m_latitudeLabel->setText("<b>" + 
	   latitudeString(m_cache.coordinate()) + "</b>");
  m_longitudeLabel->setText("<b>" + 
	   longitudeString(m_cache.coordinate()) + "</b>");
}

CoordinateTool::CoordinateTool(const Cache &cache, QWidget *parent) :
  QDialog(parent), m_cache(cache), m_iconLoader(NULL) {

  qDebug() << __FUNCTION__ << cache.waypoints().size();

  m_latitudeLabel = NULL;
  m_longitudeLabel = NULL;

  m_iconLoader = new IconLoader(64);

  /* build the coordinate edit dialog */
  setWindowTitle(tr("Edit Coordinate"));
  
  QVBoxLayout *mainLayout = new QVBoxLayout;
  mainLayout->setContentsMargins(0,0,0,0);

  SwapBox *swapBox = new SwapBox();

  /* first half of latitude */
  QWidget *hboxWidget = new QWidget;
  QHBoxLayout *hboxLayout = new QHBoxLayout;
  hboxLayout->setContentsMargins(0,0,0,0);
  hboxLayout->setSpacing(0);
  hboxWidget->setLayout(hboxLayout);

  m_latManager = new PickerManager(m_cache.coordinate(), 
				   PickerManager::Latitude);

  hboxLayout->addStretch(1);
  m_latManager->addWidgets(true, hboxLayout);
  hboxLayout->addStretch(1);
  swapBox->addWidget(hboxWidget);

  /* second half of latitude */
  hboxWidget = new QWidget;
  hboxLayout = new QHBoxLayout;
  hboxLayout->setContentsMargins(0,0,0,0);
  hboxLayout->setSpacing(0);
  hboxWidget->setLayout(hboxLayout);

  hboxLayout->addStretch(1);
  m_latManager->addWidgets(false, hboxLayout);
  hboxLayout->addStretch(1);
  swapBox->addWidget(hboxWidget);

  mainLayout->addWidget(swapBox);

  swapBox = new SwapBox();

  /* first half of longitude */
  hboxWidget = new QWidget;
  hboxLayout = new QHBoxLayout;
  hboxLayout->setContentsMargins(0,0,0,0);
  hboxLayout->setSpacing(0);
  hboxWidget->setLayout(hboxLayout);

  m_lonManager = new PickerManager(m_cache.coordinate(), 
				   PickerManager::Longitude);


  hboxLayout->addStretch(1);
  m_lonManager->addWidgets(true, hboxLayout);
  hboxLayout->addStretch(1);
  swapBox->addWidget(hboxWidget);

  /* second half of longitude */
  hboxWidget = new QWidget;
  hboxLayout = new QHBoxLayout;
  hboxLayout->setContentsMargins(0,0,0,0);
  hboxLayout->setSpacing(0);
  hboxWidget->setLayout(hboxLayout);

  hboxLayout->addStretch(1);
  m_lonManager->addWidgets(false, hboxLayout);
  hboxLayout->addStretch(1);
  swapBox->addWidget(hboxWidget);

  mainLayout->addWidget(swapBox);

  mainLayout->addStretch(1);

  /* add button box for "ok" and "cancel" */
  QDialogButtonBox *buttonBox = new QDialogButtonBox(QDialogButtonBox::Ok
				     | QDialogButtonBox::Cancel);

  connect(buttonBox, SIGNAL(accepted()), this, SLOT(accept()));
  connect(buttonBox, SIGNAL(rejected()), this, SLOT(reject()));

  connect(this, SIGNAL(accepted()), this, SLOT(acceptCoordinate()));

  mainLayout->addWidget(buttonBox);
  setLayout(mainLayout);  
}

CoordinateTool::~CoordinateTool() {
  qDebug() << __FUNCTION__;

  if(m_iconLoader)
    delete m_iconLoader;

  delete m_latManager;
  delete m_lonManager;

  // don't destroy the widget as it is owned by its parent 
  qDebug() << __FUNCTION__ << "done";
}

// returns the number num as a string with precision p and c digit mantissa
QString CoordinateTool::zeroCut(qreal num, int p, int c) {
  return QString("000").append(QString::number(num, 'f', p)).right(c+p);
}

QString CoordinateTool::latitudeString(const QGeoCoordinate &coo) {
  QString str;

  if(coo.isValid()) {
    qreal fractional, latitude = coo.latitude();
    double integral;

    if(latitude < 0) { latitude = fabs(latitude); str = tr("S");
    } else                                        str = tr("N");

    fractional = modf(latitude, &integral);
    str += " " + zeroCut(integral, 0, 2) + "\260 " +
      zeroCut(60*fractional, 3, 3) + "'";
  }
  return str;
}

QString CoordinateTool::longitudeString(const QGeoCoordinate &coo) {
  QString str;

  if(coo.isValid()) {
    qreal fractional, longitude = coo.longitude();
    double integral;

    if(longitude < 0) { longitude = fabs(longitude); str = tr("W");
    } else                                           str = tr("E");

    fractional = modf(longitude, &integral);
    str += " " + zeroCut(integral, 0, 3) + "\260 " +
      zeroCut(60*fractional, 3, 3) + "'";
  }
  return str;
}

void CoordinateTool::selectWaypoint(const QString &name) {
  qDebug() << __FUNCTION__ << name;

  QGeoCoordinate coo;
  if(name == m_cache.description())
    coo = m_cache.coordinate();
 
  foreach(Waypoint wpt, m_cache.waypoints())
    if(name == wpt.description())
      coo = wpt.coordinate();

  if(coo.isValid()) {
    // update labels
    m_latitudeLabel->setText("<b>" + latitudeString(coo) + "</b>");
    m_longitudeLabel->setText("<b>" + longitudeString(coo) + "</b>");

    emit coordinateChanged(coo);
  } else
    qDebug() << __FUNCTION__ << "could not find" << name;
}

/* todo: allow differnt layout styles */
QWidget *CoordinateTool::createWidget(const QString &title) {
  // create a groupbox for both coordinates

  QGroupBox *groupBox = new QGroupBox(title);
  QHBoxLayout *hboxLayout = new QHBoxLayout;

  QWidget *vboxWidget = new QWidget;
  QVBoxLayout *vboxLayout = new QVBoxLayout;
  vboxLayout->setContentsMargins(0,0,0,0);
  vboxLayout->setSpacing(0);
  vboxWidget->setLayout(vboxLayout);

  QComboBox *cBox = new QComboBox();
  connect(cBox, SIGNAL(activated(const QString &)), 
	  this, SLOT(selectWaypoint(const QString &)));

#if 0
  cBox->addItem(tr("New"));
  cBox->addItem(tr("Edit"));
  cBox->addItem(tr("Delete"));
  cBox->insertSeparator(3);
#endif

  cBox->addItem(m_cache.description());

  foreach(Waypoint wpt, m_cache.waypoints())
    if(wpt.coordinate().isValid()) 
      cBox->addItem(wpt.description());

  vboxLayout->addWidget(cBox);

  InfoWidget *latInfo = new InfoWidget(tr("Latitude"), 
		       latitudeString(m_cache.coordinate()));
  vboxLayout->addWidget(latInfo);
  m_latitudeLabel = latInfo->getLabel();

  InfoWidget *lonInfo = new InfoWidget(tr("Longitude"), 
		       longitudeString(m_cache.coordinate()));
  vboxLayout->addWidget(lonInfo);
  m_longitudeLabel = lonInfo->getLabel();

  hboxLayout->addWidget(vboxWidget, 1);

#if 0
  // button to edit the coordinate
  QToolButton *button = new QToolButton(); 
  //  QPixmap *icon = m_iconLoader->load("button_edit");

  QString filename = FileFinder::find("icons", "button_edit.svg");
  if(!filename.isEmpty())
    button->setIcon ( QIcon(filename) );

  connect(button, SIGNAL(clicked()), this, SLOT(editCoordinate()));
  hboxLayout->addWidget(button);
#endif

  groupBox->setLayout(hboxLayout);

  return groupBox;
}

/* let user edit the coordinate currently displayed */
void CoordinateTool::editCoordinate() {
  m_latManager->updateWidgets();
  m_lonManager->updateWidgets();
  this->exec(); 
}
