#include <QDebug>

#include <QTextDocument>
#include "gcparser.h"
#include "gcbrowser.h"

GcParser::GcParser() {
}

GcParser::~GcParser() {
}

QString GcParser::error() const {
  return m_error;
}

bool GcParser::parseFloat(const QMap<QString, QVariant> &map, const QString &key, qreal &val) {

  QMap<QString, QVariant>::const_iterator it = map.find(key);
  if(it == map.end()) {
    qDebug() << __FUNCTION__ << "Key \"" << key << "\" not found.";
    return false;
  } 

  // the value of this key should be a map
  QVariant result = it.value();
  if(QVariant::String != result.type()) {
    qDebug() << __FUNCTION__ << "Unexpected value type:" << result.type();
    return false;
  }

  val = result.toFloat();
  return true;
}

bool GcParser::parseInt(const QMap<QString, QVariant> &map, const QString &key, int &val) {

  QMap<QString, QVariant>::const_iterator it = map.find(key);
  if(it == map.end()) {
    qDebug() << __FUNCTION__ << "Key \"" << key << "\" not found.";
    return false;
  } 

  // the value of this key should be a map
  QVariant result = it.value();
  if(QVariant::String != result.type()) {
    qDebug() << __FUNCTION__ << "Unexpected value type:" << result.type();
    return false;
  }

  val = result.toInt();
  return true;
}

// additional waypoints:
// 2000: Parking, 2001: Trailhead, Stage of a Multicache, 
// 2003: Question to answer, 2004: Reference Point, Final Location

bool GcParser::parseType(const QMap<QString, QVariant> &map, 
			 const QString &key, Cache::Type &val) {
  QMap<QString, QVariant> imap;
  int i, ctid;

  if(!parseMap(map, key, imap)) {
    val = Cache::TypeUnknown;
    return false;
  }

  const struct { Cache::Type type; int ctid; } tags[] = {
    { Cache::TypeTraditional, 2 }, { Cache::TypeMulti,        3 },
    { Cache::TypeMystery,     8 }, { Cache::TypeVirtual,      4 },
    { Cache::TypeWebcam,     11 }, { Cache::TypeEvent,        6 },
    { Cache::TypeLetterbox,   5 }, { Cache::TypeEarthcache, 137 }, 
    { Cache::TypeWherigo,  1858 }, { Cache::TypeMegaEvent,  453 },
    { Cache::TypeCito,       13 }, { Cache::TypeUnknown,     -1 }
  };

  if(!parseInt(imap, "value", ctid)) {
    val = Cache::TypeUnknown;
    return false;
  }

  for(i=0;(tags[i].type != Cache::TypeUnknown) && (tags[i].ctid != ctid);i++);
  val = tags[i].type;

  return true;
}

// return a string containing everything, the string str contains 
// between the first occurances of the strings start and end
QString GcParser::subString(QString &str, const QString &start, const QString &end) {
		
  int is = str.indexOf(start) + start.length();
  int ie = str.indexOf(end, is);

  if (is == start.length()-1 || ie == -1) 
    return NULL;

  return str.mid(is, ie-is);  
}

bool GcParser::parseRating(const QMap<QString, QVariant> &map, 
			   const QString &key, Rating &rating) {
  QMap<QString, QVariant> imap;
  QString ratingStr;

  if(!parseMap(map, key, imap)) {
    rating.set(0);
    return false;
  }

  if(!parseString(imap, "text", ratingStr)) {
    rating.set(0);
    return false;
  }
  
  rating.set(ratingStr.toFloat());

  return true;
}

bool GcParser::parseContainer(const QMap<QString, QVariant> &map, 
			      const QString &key, Container &container) {
  QMap<QString, QVariant> imap;
  QString containerStr;
  int i;

  if(!parseMap(map, key, imap)) {
    container.set(Container::ContainerUnknown);
    return false;
  }

  const struct { Container::Type type; QString str; } tags[] = {
    { Container::ContainerRegular, "Regular" }, 
    { Container::ContainerSmall,     "Small" },
    { Container::ContainerMicro,     "Micro" },
    { Container::ContainerOther,     "Other" },
    { Container::ContainerNotChosen,   "Not" },  // in fact "Not Chosen"
    { Container::ContainerLarge,     "Large" },
    { Container::ContainerVirtual, "Virtual" },
    { Container::ContainerUnknown,       "?" }
  };

  if(!parseString(imap, "text", containerStr)) {
    container.set(Container::ContainerUnknown);
    return false;
  }

  container.set(Container::ContainerRegular);

  for(i=0;(tags[i].type != Container::ContainerUnknown) && 
	(tags[i].str != containerStr);i++);

  container.set(tags[i].type);

  return container.isSet();
}

bool GcParser::parseOwner(const QMap<QString, QVariant> &map, 
			      const QString &key, QString &owner) {
  QMap<QString, QVariant> imap;

  if(!parseMap(map, key, imap)) 
    return false;

  if(!parseString(imap, "text", owner))
    return false;
  
  return true;
}

bool GcParser::parseDate(const QMap<QString, QVariant> &map, 
			 const QString &key, QDate &date) {
  QString dateStr;

  if(!parseString(map, key, dateStr)) {
    date = QDate();
    return false;
  }

  QStringList dateParts = dateStr.split("/");

  // we expect month/day/year
  if(dateParts.size() != 3) return false;
  
  // qdate expects y,m,d while gc delivers m/d/y
  date = QDate(dateParts[2].toInt(), dateParts[0].toInt(), dateParts[1].toInt());

  return true;
}

bool GcParser::parseBool(const QMap<QString, QVariant> &map, const QString &key, bool &val) {

  QMap<QString, QVariant>::const_iterator it = map.find(key);
  if(it == map.end()) {
    qDebug() << __FUNCTION__ << "Key \"" << key << "\" not found.";
    return false;
  } 

  // the value of this key should be a map
  QVariant result = it.value();
  if(QVariant::Bool != result.type()) {
    qDebug() << __FUNCTION__ << "Unexpected value type:" << result.type();
    return false;
  }

  val = result.toBool();
  return true;
}

bool GcParser::parseString(const QMap<QString, QVariant> &map, const QString &key, QString &str) {

  QMap<QString, QVariant>::const_iterator it = map.find(key);
  if(it == map.end()) {
    qDebug() << __FUNCTION__ << "Key \"" << key << "\" not found.";
    return false;
  } 

  // the value of this key should be a map
  QVariant result = it.value();
  if(QVariant::String != result.type()) {
    qDebug() << __FUNCTION__ << "Unexpected value type:" << result.type();
    return false;
  }

  str = result.toString();

  return true;
}

bool GcParser::parseList(const QMap<QString, QVariant> &map, const QString &key, QList<QVariant> &list) {

  QMap<QString, QVariant>::const_iterator it = map.find(key);
  if(it == map.end()) {
    qDebug() << __FUNCTION__ << "Key \"" << key << "\" not found.";
    return false;
  } 

  QVariant result = it.value();
  if(QVariant::List != result.type()) {
    qDebug() << __FUNCTION__ << "Unexpected value type:" << result.type();
    return false;
  }

  list = result.toList();
  return true;
}

bool GcParser::parseMap(const QMap<QString, QVariant> &map, const QString &key, QMap<QString, QVariant> &nmap) {

  QMap<QString, QVariant>::const_iterator it = map.find(key);
  if(it == map.end()) {
    qDebug() << __FUNCTION__ << "Key \"" << key << "\" not found.";
    return false;
  } 

  QVariant result = it.value();
  if(QVariant::Map != result.type()) {
    qDebug() << __FUNCTION__ << "Unexpected value type:" << result.type();
    return false;
  }

  nmap = result.toMap();
  return true;
}

bool GcParser::decodeJson(const QString &data, QMap<QString, QVariant>&map) {
  bool ok;

  // json is a QString containing the data to convert
  QVariant result = Json::parse(data, ok);
  if(!ok) {
    qDebug() << __FUNCTION__ << "Json deconding failed.";
    return false;
  }

  // we expect a qvariantmap
  if(QVariant::Map != result.type()) {
    qDebug() << __FUNCTION__ << "Unexpected result type:" << result.type();
    return false;
  } 

  map = result.toMap();
  return true;
}

// stage one is just a single json encoded string stored under key "d"
bool GcParser::decodeOverview(const QString &data, QMap<QString, GcBrowser::OverviewEntry> &entryMap, int x, int y) {
  QMap<QString, QVariant> map, map2;
  GcBrowser::OverviewEntry entry;

  if(!decodeJson(data, map))
    return false;

  // the data map contains all information, we don't need "keys" and "grid"
  if(!parseMap(map, "data", map2))
    return false;

  // walk through all entries in the map
  foreach(QString key, map2.keys()) {
    QList<QVariant> list;
    QString i;
    QStringList pos = key.split(",");

    if(pos.size() == 2) {
      pos[0].remove(QChar('('));
      pos[1].remove(QChar(')'));

      entry.x = x + pos[0].toInt()/64.0;
      entry.y = y + pos[1].toInt()/64.0;
      entry.c = 1;

      if(parseList(map2, key, list)) {
	if(list.size() >= 1) {
	  int c;
	  for(c=0;c<list.size();c++) {

	    QMap<QString, QVariant> imap = list[c].toMap();
	
	    foreach(QString ikey, imap.keys()) {
	      if(!ikey.compare("i")) 
		i = imap.value(ikey).toString();
	    
	      if(!ikey.compare("n"))
		entry.n = imap.value(ikey).toString();
	    }
	    
	    // qDebug() << __FUNCTION__ << entry.x << entry.y << i << entry.n;
	    
	    // append to matching existing entry or create a new one
	    if(entryMap.contains(i)) {
	      GcBrowser::OverviewEntry old = entryMap.value(i);
	      entry.x += old.x;
	      entry.y += old.y;
	      entry.c += old.c;
	    }
	    
	    entryMap.insert(i, entry);
	  }
	} else
	  qDebug() << __FUNCTION__ << "Warning, list len < 1:" << list.size();
      } else
	qDebug() << __FUNCTION__ << "Expected list";
    } else
      qDebug() << __FUNCTION__ << "Warning, grid len != 2:" << pos.size();
  }

  return true;
}

bool GcParser::parseInfoEntry(const QMap<QString, QVariant> &map, Cache &cache) {

  /* unparsed entries:
     "subrOnly":false,  // premium only
     "li":false,        // 
     "fp":"5",          // favourtite points
  */

  QString gc, name;
  if(parseString(map, "gc", gc)) {
    cache.setName(gc);
    cache.setUrl("http://www.geocaching.com/seek/cache_details.aspx?wp=" + gc);
  }

  if(parseString(map, "name", name)) {
    // remove html tags we don't want (and don't render, anyway)
    QTextDocument textDocument;
    textDocument.setHtml(name);
    cache.setDescription(textDocument.toPlainText());
  }

  QString owner;
  if(parseOwner(map, "owner", owner))   cache.setOwner(owner);

  QDate dh;
  if(parseDate(map, "hidden", dh))   cache.setDateOfPlacement(dh);

  Rating difficulty, terrain;
  if(parseRating(map, "difficulty", difficulty))  cache.setDifficulty(difficulty.value());
  if(parseRating(map, "terrain", terrain))     cache.setTerrain(terrain.value());

  Container container;
  if(parseContainer(map, "container", container))  cache.setContainer(container.get());

  bool disabled;
  if(parseBool(map, "disabled", disabled)) cache.setAvailable(!disabled);
 
  Cache::Type type;
  if(parseType(map, "type", type)) cache.setType(type);

  QString g;
  if(parseString(map, "g", g))     cache.setGuid(g);

  qDebug() << __FUNCTION__ << cache;

  return true;
}

bool GcParser::parseStatus(const QMap<QString, QVariant> &map) {
  QString status;

  if(parseString(map, "status", status))
    return(status.compare("success") == 0);

  return false;
}

bool GcParser::decodeInfo(const QString &data, Cache &cache) {
  QMap<QString, QVariant> map;
  QList<QVariant> list;

  // assume for now that parsing failed for unknown reason
  m_error = tr("Error downloading cache info");

  if(!decodeJson(data, map))
    return false;

  if(!parseStatus(map))
    return false;

  // the data map contains all information
  if(!parseList(map, "data", list))
    return false;

  if(list.size() != 1) {
    qDebug() << __FUNCTION__ << "list size != 1: " << list.size();
    return false;
  }

  if(!parseInfoEntry(list[0].toMap(), cache))
    return false;

  m_error = QString();
  return true;
}
