#ifndef CACHEPROVIDER_H
#define CACHEPROVIDER_H

#include "cachelist.h"

#include <QQueue>
#include <QDir>
#include <QVBoxLayout>
#include <QMenuBar>

#include "cacheproviderplugin.h"

class CacheProvider  : public QObject {
  Q_OBJECT;

 public:
  enum overviewFlags {
    None        = 0x00,
    NoFound     = 0x01,
    NoOwned     = 0x02
  };

  CacheProvider(QWidget *parent = 0);
  ~CacheProvider();
  void requestOverview(const QGeoBoundingBox &, int = None);
  void requestInfo(const QString &);
  void requestDetail(const QString &);
  void start(QWidget * = 0);
  void pluginConfig(QDialog *, QVBoxLayout *);

  // functions redirected at current plugin
  QString name();
  bool busy();
  void processRequestOverview(const QGeoBoundingBox &, const int);
  void processRequestInfo(const QString &);
  void processRequestDetail(const QString &);

  void createMenu(QMenuBar *);

 public slots:
  void next();
  void done();

 private slots:
  void emitReplyOverview(const CacheList &);
  void emitReplyInfo(const Cache &);
  void emitReplyDetail(const Cache &);
  void emitReplyError(const QString &);
  void emitReload();
  void emitNotifyBusy(bool);
  void providerSelected(QAction *);

 signals:
  void replyOverview(const CacheList &);
  void replyInfo(const Cache &);
  void replyDetail(const Cache &);
  void replyError(const QString &);    // any request can emit a error
  void reload();
  void notifyBusy(bool);
  void waypointsChanged();

 public:

  // a request entry is being kept in the request queue
  class RequestEntry {    
    
  public: 
    enum RequestType { Overview, Info, Detail };
    RequestEntry(const RequestType &, const QString &);
    RequestEntry(const RequestType &, const QGeoBoundingBox &, const int);
    int flags();
    RequestType type();
    QGeoBoundingBox bbox();
    QString cache();
    void set(const QGeoBoundingBox &, const int);
    void set(const QString &);

  private:
    RequestType m_type;
    QGeoBoundingBox m_bbox;
    QString m_cache;
    int m_flags;
  };

  class RequestQueue : public QQueue<RequestEntry *> {
  private:
    CacheProvider *m_cacheProvider;

  public:
    RequestQueue(CacheProvider *);
    bool add(const RequestEntry::RequestType &, const QString &);
    bool add(const RequestEntry::RequestType &, const QGeoBoundingBox &, const int);
    RequestEntry::RequestType type();
    void done();
    void next();
    void restart();
  };
  
  RequestEntry::RequestType type();

 private:
  void loadPluginsInDir(QDir &);

  // queue of pending requests
  RequestQueue *m_pending;

  // list of available cache provider plugins
  QList <CacheProviderPlugin *> m_cacheProviderPluginList;
  CacheProviderPlugin *m_currentPlugin;
  QWidget *m_parent;
};

#endif // CACHEPROVIDER_H
