#include <QDebug>

#include <QLabel>
#include <QGroupBox>
#include <math.h>

#include "navigationwindow.h"
#include "compass.h"
#include "swapbox.h"

// a hbox with two strings, one left aligned and one right aligned
class InfoWidget : public QWidget {
public:
  InfoWidget(const QString &leftStr = "",
	     const QString &rightStr = "",
	     QWidget *parent = 0) : QWidget(parent) {

    QHBoxLayout *layout = new QHBoxLayout;
    layout->setContentsMargins(0,0,0,0);
    layout->setSpacing(0);

    m_left = new QLabel(leftStr + ":");
    layout->addWidget(m_left);
    layout->addStretch();
    m_right = new QLabel("<b>" + rightStr + "</b>");
    layout->addWidget(m_right);

    setLayout(layout);
  }

  void setInfo(const QString &info) {
    m_right->setText("<b>" + info + "</b>");
  }

private:
  QLabel *m_left, *m_right;
};

NavigationWindow::NavigationWindow(const Cache &cache, 
	   LocationProvider *locationProvider, QWidget *parent) : 
  CustomWindow("NavigationWindow", parent), m_cache(cache) {

  setWindowTitle(tr("Navigation"));

  SwapBox *sbox = new SwapBox();

  // left/top: the compass
  Compass *compass = new Compass(m_cache, this);
  sbox->addWidget(compass);

  // right/bottom: the textual info
  QWidget *vbox = new QWidget;
  QVBoxLayout *layout = new QVBoxLayout;
  layout->setContentsMargins(0,0,0,0);
  layout->setSpacing(0);

  QGroupBox *groupBox = new QGroupBox(tr("Distance"));
  QVBoxLayout *ivbox = new QVBoxLayout;
  m_distance = new QLabel("");
  ivbox->addWidget(m_distance, 0, Qt::AlignHCenter);
  groupBox->setLayout(ivbox);
  layout->addWidget(groupBox);
  layout->addStretch();

  groupBox = new QGroupBox(tr("Current coordinate"));
  ivbox = new QVBoxLayout;
  m_latitude = new InfoWidget(tr("Latitude"));
  ivbox->addWidget(m_latitude);
  m_longitude = new InfoWidget(tr("Longitude"));
  ivbox->addWidget(m_longitude);
  groupBox->setLayout(ivbox);
  layout->addWidget(groupBox);
  layout->addStretch();

  groupBox = new QGroupBox(tr("Cache coordinate"));
  ivbox = new QVBoxLayout;
  ivbox->addWidget(new InfoWidget(tr("Latitude"), 
		 latitudeString(m_cache.coordinate())));
  ivbox->addWidget(new InfoWidget(tr("Longitude"),
  	         longitudeString(m_cache.coordinate())));
  groupBox->setLayout(ivbox);
  layout->addWidget(groupBox);
  layout->addStretch();


  vbox->setLayout(layout);
  sbox->addWidget(vbox);

  setCentralWidget(sbox);

  // send position/azimuth updates to compass widget
  connect(locationProvider, SIGNAL(positionUpdated(const QGeoPositionInfo &)), 
	  compass, SLOT(setPosition(const QGeoPositionInfo &)));
  connect(locationProvider, SIGNAL(positionUpdated(const QCompassReading *)), 
	  compass, SLOT(setPosition(const QCompassReading *)));

  // and process position locally
  connect(locationProvider, SIGNAL(positionUpdated(const QGeoPositionInfo &)), 
	  this, SLOT(setPosition(const QGeoPositionInfo &)));
}

NavigationWindow::~NavigationWindow() {
  qDebug() << __FUNCTION__;
}

void NavigationWindow::setPosition(const QGeoPositionInfo &pos) {
  qreal dist = pos.coordinate().distanceTo(m_cache.coordinate());

  m_distance->setText("<b>" + distanceString(dist) + "</b>");
  m_latitude->setInfo(latitudeString(pos.coordinate()));
  m_longitude->setInfo(longitudeString(pos.coordinate()));
}

QString NavigationWindow::zeroCut(qreal num, int p, int c) {
  return QString("000").append(QString::number(num, 'f', p)).right(c+p);
}

QString NavigationWindow::latitudeString(const QGeoCoordinate &coo) {
  QString str;

  if(coo.isValid()) {
    qreal fractional, latitude = coo.latitude();
    double integral;

    if(latitude < 0) { latitude = fabs(latitude); str = tr("S");
    } else                                        str = tr("N");

    fractional = modf(latitude, &integral);
    str += " " + zeroCut(integral, 0, 2) + "\260 " +
      zeroCut(60*fractional, 3, 3) + "'";
  }
  return str;
}

QString NavigationWindow::longitudeString(const QGeoCoordinate &coo) {
  QString str;

  if(coo.isValid()) {
    qreal fractional, longitude = coo.longitude();
    double integral;

    if(longitude < 0) { longitude = fabs(longitude); str = tr("W");
    } else                                           str = tr("E");

    fractional = modf(longitude, &integral);
    str += " " + zeroCut(integral, 0, 3) + "\260 " +
      zeroCut(60*fractional, 3, 3) + "'";
  }
  return str;
}

QString NavigationWindow::distanceString(qreal dist) {
  // current we only support metric units
  
#if 0  
  /* 1 mil = 1760 yd = 5280 ft ... */
  if(dist<0.018)      snprintf(str, len, "%.1f ft", dist*5280.0);
  else if(dist<0.055) snprintf(str, len, "%.1f yd", dist*1760.0);
  else if(dist<0.55)  snprintf(str, len, "%.0f yd", dist*1760.0);
  else if(dist<10.0)  snprintf(str, len, "%.2f mi", dist);
  else if(dist<100.0) snprintf(str, len, "%.1f mi", dist);
  else                snprintf(str, len, "%.0f mi", dist);
#endif

  QString str;
  if(dist<10)          str = QString::number(dist, 'f', 2) + " m";
  else if(dist<100)    str = QString::number(dist, 'f', 1) + " m";
  else if(dist<1000)   str = QString::number(dist, 'f', 0) + " m";
  else if(dist<100000) str = QString::number(dist/1000, 'f', 1) + " km";
  else                 str = QString::number(dist/1000, 'f', 0) + " km";

  return str;
}


