/*
  cacheprovider, class answering requests for caches

  The cacheprovider handles requests asynchronously and 
  internally uses a queue to store requests. It also drops
  unhandled pending requests if a newer request of the same
  time comes in
*/

#include <QDebug>

// to load plugins
#include <QApplication>
#include <QPluginLoader>
#include <QDir>

#include "cacheproviderplugin.h"
#include "cacheprovider.h"

CacheProvider::RequestEntry::RequestEntry(const RequestType &type, const QString &cache) {

  Q_ASSERT((type == Info) || (type == Detail));

  this->m_type = type;
  this->m_cache = cache;
}

CacheProvider::RequestEntry::RequestEntry(const RequestType &type, const QGeoBoundingBox &bbox) {

  Q_ASSERT(type == Overview);

  this->m_type = type;
  this->m_bbox = bbox;
}

CacheProvider::RequestEntry::RequestType CacheProvider::RequestEntry::type() {
  return this->m_type;
}

QGeoBoundingBox CacheProvider::RequestEntry::bbox() {
  return this->m_bbox;
}

QString CacheProvider::RequestEntry::cache() {
  return this->m_cache;
}

void CacheProvider::RequestEntry::set(const QString &cache) {
  this->m_cache = cache;
}

void CacheProvider::RequestEntry::set(const QGeoBoundingBox &bbox) {
  this->m_bbox = bbox;  
}

CacheProvider::RequestQueue::RequestQueue(CacheProvider *cacheProvider) {
  this->m_cacheProvider = cacheProvider;
}

bool CacheProvider::RequestQueue::add(const RequestEntry::RequestType &type, const QString &cache) {

  // ignore detail requests if there's already one being processed. This
  // prevents multiple stacked windows from opening on maemo5
  if(size() && (type == RequestEntry::Detail) && (head()->type() == RequestEntry::Detail)) {
    qDebug() << __FUNCTION__ << "supressing addional Detail request";
    return false;
  }

  // check if there's already the same type of request in
  // queue and overwrite that one
  if(size() > 1) {
    for(int i=1;i<size();i++) {
      if(at(i)->type() == type) {
	at(i)->set(cache);
	return false;
      }      
    }
  }

  enqueue(new RequestEntry(type, cache));
  return true;
}

bool CacheProvider::RequestQueue::add(const RequestEntry::RequestType &type, const QGeoBoundingBox &bbox) {

  // check if there's already the same type of request in
  // queue and overwrite that one
  if(size() > 1) {
    for(int i=1;i<size();i++) {
      if(at(i)->type() == type) {
	at(i)->set(bbox);
	return false;
      }      
    }
  }

  enqueue(new RequestEntry(type, bbox));
  return true;
}

CacheProvider::RequestEntry::RequestType CacheProvider::RequestQueue::type() {
  return head()->type();
}

void CacheProvider::RequestQueue::done() {
  qDebug() << __FUNCTION__;
  delete dequeue();
}

void CacheProvider::RequestQueue::next() {

  if(!empty()) {

    // process topmost entry
    switch(type()) {
    case RequestEntry::Overview:
      qDebug() << __FUNCTION__ << "Pending Overview request";
      m_cacheProvider->processRequestOverview(head()->bbox()); 
      break;

    case RequestEntry::Info:
      qDebug() << __FUNCTION__ << "Pending Info request";
      m_cacheProvider->processRequestInfo(head()->cache()); 
      break; 

    case RequestEntry::Detail:
      qDebug() << __FUNCTION__ << "Pending Detail request";
      m_cacheProvider->processRequestDetail(head()->cache()); 
      break; 

    default:
      Q_ASSERT(0);
      break;
    }    
  } else
    qDebug() << __FUNCTION__ << "No pending request";
}

void CacheProvider::RequestQueue::restart() {
  // restart if queue holds one entry and if the cache provider
  // is not busy doing its own stuff (e.g. logging in)
  if((size() == 1) && !m_cacheProvider->busy()) 
    next();
}

void CacheProvider::loadPluginsInDir(QDir &pluginsDir) {

  if(!pluginsDir.cd("plugins")) return;
  if(!pluginsDir.cd("cacheprovider")) return;
  
  qDebug() << __FUNCTION__ << "Loading plugins from:" << pluginsDir;

  // load all available plugins
  foreach (QString fileName, pluginsDir.entryList(QDir::Files)) {
    qDebug() << __FUNCTION__ << fileName;
    QPluginLoader loader(pluginsDir.absoluteFilePath(fileName));
    QObject *plugin = loader.instance();
    if (plugin) {
      CacheProviderPlugin *cacheProviderPlugin = 
	qobject_cast<CacheProviderPlugin *>(plugin);

      m_cacheProviderPluginList.append(cacheProviderPlugin);

      qDebug() << __FUNCTION__ << "Plugin name:" << cacheProviderPlugin->name();
    }
  }
}
  
CacheProvider::CacheProvider() : m_currentPlugin(NULL) {
  QDir pluginsDir = QDir(QApplication::applicationDirPath());

  // try to load plugins from local path first
#if defined(Q_OS_WIN)
  if (pluginsDir.dirName().toLower() == "debug" || 
      pluginsDir.dirName().toLower() == "release")
    pluginsDir.cdUp();
#endif
  loadPluginsInDir(pluginsDir);

#ifdef LIBDIR
  if(m_cacheProviderPluginList.isEmpty()) {
    // if no local plugins were found -> try the intallation path
    pluginsDir = QDir(QDir::toNativeSeparators(LIBDIR));  
    pluginsDir.cd(APPNAME);
    loadPluginsInDir(pluginsDir);
  }
#endif

  this->m_pending = new RequestQueue(this);

  // activate first plugin if available
  if(!m_cacheProviderPluginList.isEmpty())
    this->m_currentPlugin = m_cacheProviderPluginList[0];
}

CacheProvider::~CacheProvider() {
  qDebug() << __FUNCTION__;
  delete this->m_pending;
}

void CacheProvider::requestOverview(const QGeoBoundingBox &area) {
  qDebug() << __PRETTY_FUNCTION__ << "from" << area.topLeft() << 
    " to " << area.bottomRight();

  if(m_pending->add(RequestEntry::Overview, area))
    m_pending->restart();
}

void CacheProvider::requestInfo(const QString &cache) {
  qDebug() << __PRETTY_FUNCTION__ << cache;

  if(m_pending->add(RequestEntry::Info, cache))
    m_pending->restart();
}

void CacheProvider::requestDetail(const QString &cache) {
  qDebug() << __PRETTY_FUNCTION__ << cache;

  if(m_pending->add(RequestEntry::Detail, cache))
    m_pending->restart();
}

void CacheProvider::next() {
  this->m_pending->next();
}

void CacheProvider::done() {
  this->m_pending->done();
}

CacheProvider::RequestEntry::RequestType CacheProvider::type() {
  return this->m_pending->type();
}

// the following functions are called by the plugin in order to 
// let the CacheProvider emit a signal
void CacheProvider::emitReplyOverview(const CacheList &cacheList) {
  emit replyOverview(cacheList);
}

void CacheProvider::emitReplyInfo(const Cache &cache) {
  emit replyInfo(cache);
}

void CacheProvider::emitReplyDetail(const Cache &cache) {
  emit replyDetail(cache);
}

void CacheProvider::emitReplyError(const QString &msg) {
  emit replyError(msg);
}

void CacheProvider::emitSettingsChanged() {
  emit settingsChanged();
}

void CacheProvider::emitBusy(bool on) {
  emit busy(on);
}

QString CacheProvider::name() {
  if(!m_currentPlugin) return("<none>");
  return m_currentPlugin->name();
}

bool CacheProvider::busy() {
  if(!m_currentPlugin) return(true);
  return m_currentPlugin->busy();
}
 
void CacheProvider::processRequestOverview(const QGeoBoundingBox &bbox) {
  if(m_currentPlugin)
    m_currentPlugin->processRequestOverview(bbox);
}

void CacheProvider::processRequestInfo(const QString &id) {
  if(m_currentPlugin)
    m_currentPlugin->processRequestInfo(id);
}

void CacheProvider::processRequestDetail(const QString &id) {
  if(m_currentPlugin)
    m_currentPlugin->processRequestDetail(id);
}

void CacheProvider::start() {
  foreach(CacheProviderPlugin *plugin, m_cacheProviderPluginList) 
    plugin->setCacheProvider(this);
}

void CacheProvider::pluginConfig(QDialog *parent, QVBoxLayout *box) {
  foreach(CacheProviderPlugin *plugin, m_cacheProviderPluginList)
    plugin->createConfig(parent, box);
}
