// Filename: mainwindow.cpp

#include <QDebug>
#include <QMainWindow>
#include <QMessageBox>

#include "mainwindow.h"
#include "mapwidget.h"
#include "mapdialog.h"
#include "iconloader.h"  
#include <QIcon>

#ifdef Q_WS_MAEMO_5
#include <QMaemo5InformationBox>
#include <QtGui/QX11Info>
#include <X11/Xlib.h>
#include <X11/Xatom.h>

void grabZoomKeys(QWidget *window) {
  if(!window->winId()) return;
  
  unsigned long val = 1;
  Atom atom = XInternAtom(QX11Info::display(), "_HILDON_ZOOM_KEY_ATOM", False);
  if(!atom) return;
  
  XChangeProperty (QX11Info::display(), window->winId(), atom, 
		   XA_INTEGER, 32, PropModeReplace,
		   reinterpret_cast<unsigned char *>(&val),  1);
}
#else
#include <QStatusBar>
#include <QVBoxLayout>
#endif

void MainWindow::createMenu() {
  QMenu *menu;
  QAction *action;

#ifndef Q_WS_MAEMO_5
  menu = new QMenu(tr("&File"), this);
  action = new QAction(tr("E&xit"), this);
  action->setShortcuts(QKeySequence::Quit);
  connect(action, SIGNAL(triggered()), this, SLOT(close()));
  menu->addAction(action);
  menuBar()->addMenu(menu);
#endif
  
  menu = new QMenu(tr("&Map"), this);
  action = new QAction(tr("Map &Settings"), this);
  connect(action, SIGNAL(triggered()), this, SLOT(mapdialog()));
  menu->addAction(action);
  menuBar()->addMenu(menu);

  menu = new QMenu(tr("&Help"), this);
  action = new QAction(tr("&About"), this);
  connect(action, SIGNAL(triggered()), this, SLOT(about()));
  menu->addAction(action);
  menuBar()->addMenu(menu);
}

void MainWindow::mapdialog() {
  qDebug() << __FUNCTION__;

  MapDialog dialog(m_mapWidget, this);
  dialog.exec(); 
}

void MainWindow::about() {
  QMessageBox::about(this, tr("About CacheMe"),
	     tr("CacheMe") + " " + tr("Version") + " " + VERSION + "\n" + 
	     tr("Build date:") + " " + __DATE__ + "\n" +
	     tr("(c) 2011 by Till Harbaum <till@harbaum.org>") + "\n\n" +
	     tr("CacheMe is an online geocaching application.")
	     );
}


MainWindow::MainWindow(LocationProvider *locationProvider, QWidget *parent) : 
  QMainWindow(parent), m_locationProvider(locationProvider) {

  qDebug() << __FUNCTION__ << "Cache provider " << m_cacheProvider.name();

  setWindowTitle(tr("CacheMe"));

  IconLoader iconLoader(48);
  QPixmap *icon = iconLoader.load(APPNAME);
  if(icon) setWindowIcon(QIcon(*icon));

#ifdef Q_WS_MAEMO_5
  setAttribute(Qt::WA_Maemo5AutoOrientation, true);
  grabZoomKeys(this);
#endif

  createMenu();

  // search for service providers
  QStringList list = QGeoServiceProvider::availableServiceProviders();
  for(int i=0;i<list.size();i++)
    qDebug() << __FUNCTION__ << "Service provider: " << list[i];

  // there needs to be at least one geo service provider
  Q_ASSERT(list.size() > 0);

  QSettings settings;
  settings.beginGroup("Map");
  QString serviceProviderStr = settings.value("Service", "nokia").toString();
  settings.endGroup();

  this->m_serviceProvider = new QGeoServiceProvider(serviceProviderStr);
  if(!this->m_serviceProvider->mappingManager())
    this->m_serviceProvider = new QGeoServiceProvider("nokia");

  this->m_mapWidget = new MapWidget(this->m_serviceProvider->mappingManager());

  // make sure cache updates are processed
  QObject::connect(this->m_locationProvider, SIGNAL(positionUpdated(const QGeoPositionInfo &)), 
		   this->m_mapWidget, SLOT(positionUpdated(const QGeoPositionInfo &)));

  QObject::connect(&this->m_cacheProvider, SIGNAL(replyOverview(const CacheList &)), 
		   this, SLOT(updateCaches(const CacheList &)));

  QObject::connect(&this->m_cacheProvider, SIGNAL(replyInfo(const Cache &)), 
		   this->m_mapWidget, SLOT(showBubble(const Cache &)));

  QObject::connect(&this->m_cacheProvider, SIGNAL(replyError(const QString &)), 
		   this, SLOT(displayError(const QString &)));

  QObject::connect(this->m_mapWidget, SIGNAL(mapChanged()),
		   this, SLOT(reloadCaches()));

  QObject::connect(this->m_mapWidget, SIGNAL(showMessage(const QString &)),
		   this, SLOT(statusMessage(const QString &)));

  QObject::connect(this, SIGNAL(increasePressed()),
		   this->m_mapWidget, SLOT(zoomIn()));

  QObject::connect(this, SIGNAL(decreasePressed()),
		   this->m_mapWidget, SLOT(zoomOut()));

  // connect to map to handle clicks on cache icons
  QObject::connect(this->m_mapWidget, SIGNAL(cacheClicked(const QString &)),
		   this, SLOT(selectCache(const QString &)));

  QGraphicsScene *scene = new QGraphicsScene(this);
  m_view = new QGraphicsView( scene );
  m_view->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
  m_view->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
  m_view->setVisible(true);
  m_view->setInteractive(true);

  scene->addItem(m_mapWidget);

  setCentralWidget(m_view);

#ifndef Q_WS_MAEMO_5
  // on maemo5 we use QMaemo5InformationBox
  setStatusBar(new QStatusBar(this));
#endif

  settings.beginGroup("MainWindow");
  resize(settings.value("size", QSize(640, 480)).toSize());
  move(settings.value("pos", QPoint(200, 200)).toPoint());
  settings.endGroup();
}

MainWindow::~MainWindow() {
  qDebug() << __FUNCTION__;

  QSettings settings;
  settings.beginGroup("MainWindow");
  settings.setValue("size", size());
  settings.setValue("pos", pos());
  settings.endGroup();
  
  delete m_serviceProvider;
}

void MainWindow::resizeEvent(QResizeEvent*) {
  m_view->setSceneRect(QRectF(QPointF(0.0, 0.0), m_view->size()));
  m_mapWidget->resize(m_view->size());
}

void MainWindow::keyPressEvent(QKeyEvent* event) {
  // check if device is in portrait mode
  bool portrait = size().height() > size().width();

  switch (event->key()) {
  case Qt::Key_F7:
    if(portrait) emit decreasePressed();
    else         emit increasePressed();
    event->accept();
    break;
    
  case Qt::Key_F8:
    if(portrait) emit increasePressed();
    else         emit decreasePressed();
    event->accept();
    break;
  }
  QWidget::keyPressEvent(event);
}

void MainWindow::selectCache(const QString &name) {
  // ask CacheProvider for details about this cache
  this->m_cacheProvider.requestInfo(name);
}

void MainWindow::reloadCaches() {
  // request new caches for the maps new viewport
  m_cacheProvider.requestOverview(this->m_mapWidget->viewport());
}

void MainWindow::updateCaches(const CacheList &cacheList) {
#ifndef Q_WS_MAEMO_5
  // check if it's actually one of the error messages that's still
  // displayed and remove it in that case
  if(statusBar()->property("error").toBool())  
    statusBar()->clearMessage();
#endif

  m_message = "";
  this->m_mapWidget->updateCaches(cacheList);
}

void MainWindow::statusMessage(const QString &message, int time) {
#ifdef Q_WS_MAEMO_5
  QMaemo5InformationBox::information(this, message);
#else
  statusBar()->setProperty("error", false);  
  statusBar()->showMessage(message, time);
#endif
}

void MainWindow::displayError(const QString &message) {
  if(m_message != message) {
    statusMessage(message, 0);
    m_message = message;

#ifndef Q_WS_MAEMO_5
    statusBar()->setProperty("error", true );
#endif
  }
}
