#include <QDebug>

#include <QGeoServiceProvider>
#include <QGraphicsSceneMouseEvent>
#include <QGeoMapOverlay>
#include <QSettings>

#include "mapwidget.h"

void MapWidget::updateZoomButtons() {
#if(QTM_VERSION >= 0x010200)
  this->m_mapOverlay->enableZoomButtons(
		zoomLevel() < maximumZoomLevel(),
		zoomLevel() > minimumZoomLevel());
  
  update();
#endif
}

MapWidget::MapWidget(QGeoMappingManager *mgr) : 
  QGraphicsGeoMap(mgr), m_bubble(NULL), m_posIsValid(false), m_gpsMarker(NULL) {

  // try to get default from qsettings, otherwise start at berlin's zoo
  QSettings settings;
  settings.beginGroup("Map");
  setCenter(QGeoCoordinate(
           settings.value("Center/Latitude", 52.514).toFloat(),
	   settings.value("Center/Longitude", 13.3611).toFloat()));
  setZoomLevel(settings.value("ZoomLevel", 15).toInt());
  m_iconSize = settings.value("IconSize", 48).toInt();  
  m_gpsLocked = settings.value("GpsLocked", false).toBool();
  settings.endGroup();
  m_iconLoader = new IconLoader(m_iconSize);

#if(QTM_VERSION >= 0x010200)
  // create the control overlays
  this->m_mapOverlay = new MapOverlay(this);
  addMapOverlay( this->m_mapOverlay );

  // and update it to reflect restored map state
  updateZoomButtons();
  if(m_gpsLocked)
    this->m_mapOverlay->changeGpsButton(true);
#endif

  // install timer to delay cache reload requests
  this->m_timer = new QTimer(this);
  this->m_timer->setSingleShot(true);
  QObject::connect(this->m_timer, SIGNAL(timeout()),
		   this, SLOT(reloadTimerExpired()));

  // connect to map signals to be able to handle viewport changes
  QObject::connect(this, SIGNAL(centerChanged(const QGeoCoordinate &)),
		   this, SLOT(centerChangedEvent(const QGeoCoordinate &)));
  QObject::connect(this, SIGNAL(zoomLevelChanged(qreal)),
		   this, SLOT(zoomLevelChangedEvent(qreal)));
}

MapWidget::~MapWidget() {
  qDebug() << __FUNCTION__;

  QSettings settings;
  settings.beginGroup("Map");
  settings.setValue("Center/Latitude", center().latitude());
  settings.setValue("Center/Longitude", center().longitude());
  settings.setValue("ZoomLevel", zoomLevel());
  settings.setValue("IconSize", m_iconSize);
  settings.setValue("GpsLocked", m_gpsLocked);
  settings.endGroup();

  qDebug() << __FUNCTION__ << center() << zoomLevel();

  delete m_iconLoader;
  delete m_timer;
}

#if (QTM_VERSION < 0x010200)
// early qtm versions lack viewport()
QGeoBoundingBox MapWidget::viewport() const {
  qDebug() << __FUNCTION__;

  return(QGeoBoundingBox(screenPositionToCoordinate(QPointF(0,0)),
    screenPositionToCoordinate(QPointF(size().width(),size().height()))));
}
#endif

// ------------ slots to handle overlay button clicks --------------
void MapWidget::zoomIn() {
  if(zoomLevel() < maximumZoomLevel()) {
    setZoomLevel(zoomLevel()+1);
    updateZoomButtons();
  }
}

void MapWidget::zoomOut() {
  if(zoomLevel() > minimumZoomLevel()) {
    setZoomLevel(zoomLevel()-1);
    updateZoomButtons();
  }
}

void MapWidget::gpsFollow() {
#if(QTM_VERSION >= 0x010200)
  m_gpsLocked = true;
  this->m_mapOverlay->changeGpsButton(true);  
  update();
#endif
}

void MapWidget::resizeEvent(QGraphicsSceneResizeEvent *event) {
  QGraphicsGeoMap::resizeEvent(event);
  reload();
}

void MapWidget::showEvent(QShowEvent *) {
  reload();
}

void MapWidget::zoomLevelChangedEvent(qreal) {
  reload();
}

void MapWidget::centerChangedEvent(const QGeoCoordinate &) {
  reload();
}

void MapWidget::hideBubble() {
  // kill any existing bubble
  if(m_bubble) {
    removeMapObject(m_bubble);
    delete m_bubble;
    m_bubble = NULL;
  }
}

void MapWidget::showBubble(const Cache &cache) {
  qDebug() << __FUNCTION__ << cache.name();

  hideBubble();
  m_bubble = new MapBubble(this, cache);
  addMapObject(m_bubble);
}

void MapWidget::mousePressEvent(QGraphicsSceneMouseEvent* event) {
#if(QTM_VERSION >= 0x010200)
  if(this->m_mapOverlay->mousePress(event->pos()))
    update();

  if(this->m_mapOverlay->isInside(event->pos()))
    m_downPos = QPointF(0,0);
  else
#endif
    m_downPos = event->pos();

  m_dragging = false;
}

void MapWidget::mouseReleaseEvent(QGraphicsSceneMouseEvent* event) {
#if(QTM_VERSION >= 0x010200)
  if(this->m_mapOverlay->mouseRelease(event->pos()))
    update();
#endif

  if(m_dragging && m_gpsLocked) {
#if(QTM_VERSION >= 0x010200)
    m_gpsLocked = false;
    this->m_mapOverlay->changeGpsButton(false);
    update();
#endif
  }

  if(m_downPos.isNull() || m_dragging)
    return;

  // check if we clicked a cache
  bool clickedBubble = false;
  QString clickedCache;

  QList<QGeoMapObject*> objects = mapObjectsAtScreenPosition(m_downPos);
  if (objects.length() > 0) {
    for (int i = objects.length()-1; i >= 0; i--) {
      if ((objects[i]->objectName() == "cache") && clickedCache.isEmpty())
	clickedCache = objects[i]->property("name").toString();
      
      if (objects[i]->objectName() == "bubble") 
	clickedBubble = true;
    }
  }

  if(!clickedBubble)
    hideBubble();

  if(!clickedCache.isEmpty()) 
    emit cacheClicked(clickedCache);
}

void MapWidget::mouseMoveEvent(QGraphicsSceneMouseEvent* event) {
#if(QTM_VERSION >= 0x010200)
  if(this->m_mapOverlay->mouseMove(event->pos()))
    update();
#endif

  if(m_downPos.isNull())
    return;

  if(!m_dragging) {
    if((m_downPos - event->pos()).manhattanLength() < DRAG_FUZZ) 
      return;

    // dragged more than DRAG_FUZZ pixels: start actual drag
    m_dragging = true;
    QPointF dp(m_downPos - event->pos());
    pan(dp.x(), dp.y());
    return;
  }

  QPointF d(event->lastPos() - event->pos());
  pan(d.x(), d.y());
}

MapWidget::MapCacheObject::MapCacheObject(IconLoader *loader, const Cache &cache) :
  QGeoMapPixmapObject(cache.coordinate(), 
		      QPoint(-loader->size()/2, -loader->size()/2),
		      *loader->load(cache)) {
  
  setObjectName("cache");
  setProperty("name", cache.name());
  setProperty("description", cache.description());
};

// the pixmap object seems to actually check for clicks onto the bitmap. thus
// we only check for the click being withing the entire pixmap size. furthermore
// the pixmap precise stuff doesn't seem to work at all on the n900/maemo
bool MapWidget::MapCacheObject::contains(const QGeoCoordinate &coo) const {
  return boundingBox().contains(coo);
}

void MapWidget::updateCaches(const CacheList &cacheList) {
  // save cache if existing bubble
  Cache bubbleCache;
  if(m_bubble) bubbleCache = m_bubble->cache();

  // save location of gps marker
  QGeoCoordinate gpsCoo;
  if(m_gpsMarker) gpsCoo = m_gpsMarker->coordinate();

  // remove existing objects (incl. a possible bubble)
  clearMapObjects();
  m_bubble = NULL;
  m_gpsMarker = NULL;

  // draw all caches
  QList<Cache>::const_iterator i;
  for( i = cacheList.begin(); i != cacheList.end(); ++i ) 
    addMapObject(new MapCacheObject(m_iconLoader, *i));

  // restore previous bubble
  if(bubbleCache.coordinate().isValid()) {
    // The ugly part: gc.com is randomizing coordinates and
    // the new coordinates used to draw the icon are likely
    // different from the ones saved from the previous bubble
    // position. So we search for the bubble cache within the
    // new cache list and use it's coordinates instead
    
    bool coordinateUpdated = false;
    for( i = cacheList.begin(); i != cacheList.end(); ++i ) {
      if(i->name() == bubbleCache.name()) {
	bubbleCache.setCoordinate(i->coordinate());
	coordinateUpdated = true;
      }
    }

    // only redraw bubble if the cache is still part of the
    // new cache list and if it's cache is still on-screen
    if(coordinateUpdated && 
       viewport().contains(bubbleCache.coordinate()))
      showBubble(bubbleCache);
  }

  // restore gps marker
  if(gpsCoo.isValid()) {
    m_gpsMarker = new QGeoMapPixmapObject(gpsCoo, 
	  QPoint(-m_iconLoader->size()/2, -m_iconLoader->size()/2),
	  *m_iconLoader->load("gps_marker"));
    addMapObject(m_gpsMarker);
  }
}

void MapWidget::reloadTimerExpired() {
  qDebug() << __FUNCTION__;

  // check if viewport actually changed
  if(m_currentViewport != viewport()) {
    emit mapChanged();
    m_currentViewport = viewport();
  }
}

void MapWidget::reload() {
  // start the timer if it's not already running, re-start from zero otherwise
  if(!this->m_timer->isActive())
    this->m_timer->start(1000);
  else
    this->m_timer->setInterval(1000);
}

void MapWidget::positionUpdated(const QGeoCoordinate &coo) {

  // change state of gps button to reflect gps availability
  if(m_posIsValid != coo.isValid()) {
#if(QTM_VERSION >= 0x010200)
    this->m_mapOverlay->enableGpsButton(coo.isValid());
    update();
#endif
    m_posIsValid = coo.isValid();
  }

  // re-center map if gpsLock is active
  if(coo.isValid() && m_gpsLocked) 
    setCenter(coo);

  // draw gps marker
  if(coo.isValid()) {
    if(m_gpsMarker) {
      // move existing marker
      m_gpsMarker->setCoordinate(coo);
    } else {
      // create new marker
      m_gpsMarker = new QGeoMapPixmapObject(coo, 
	    QPoint(-m_iconLoader->size()/2, -m_iconLoader->size()/2),
	    *m_iconLoader->load("gps_marker"));
      addMapObject(m_gpsMarker);
    }
  } else {
    // remove existing marker
    removeMapObject(m_gpsMarker);
    delete m_gpsMarker;
    m_gpsMarker = NULL;
  }
}
