/*
  cacheprovider, virtual class answering requests for caches

  The cacheprovider handles requests asynchronously and 
  internally uses a queue to store requests. It also drops
  unhandled pending requests if a newer request of the same
  time comes in
*/

#include <QDebug>

#include "cacheprovider.h"

CacheProvider::RequestEntry::RequestEntry(const RequestType &type, const QString &cache) {

  Q_ASSERT((type == Info) || (type == Detail));

  this->m_type = type;
  this->m_cache = cache;
}

CacheProvider::RequestEntry::RequestEntry(const RequestType &type, const QGeoBoundingBox &bbox) {

  Q_ASSERT(type == Overview);

  this->m_type = type;
  this->m_bbox = bbox;
}

CacheProvider::RequestEntry::RequestType CacheProvider::RequestEntry::type() {
  return this->m_type;
}

QGeoBoundingBox CacheProvider::RequestEntry::bbox() {
  return this->m_bbox;
}

QString CacheProvider::RequestEntry::cache() {
  return this->m_cache;
}

CacheProvider::RequestQueue::RequestQueue(CacheProvider *cacheProvider) {
  this->m_cacheProvider = cacheProvider;
}

void CacheProvider::RequestQueue::add(const RequestEntry::RequestType &type, const QString &cache) {
  enqueue(new RequestEntry(type, cache));
}

void CacheProvider::RequestQueue::add(const RequestEntry::RequestType &type, const QGeoBoundingBox &bbox) {
  qDebug() << __PRETTY_FUNCTION__;
  enqueue(new RequestEntry(type, bbox));
};

CacheProvider::RequestEntry::RequestType CacheProvider::RequestQueue::type() {
  return head()->type();
}

void CacheProvider::RequestQueue::done() {
  qDebug() << __PRETTY_FUNCTION__;
  delete dequeue();
}

void CacheProvider::RequestQueue::next() {
  if(!empty()) {

    // process topmost entry
    switch(type()) {
    case RequestEntry::Overview:
      qDebug() << __FUNCTION__ << "Pending Overview request";
      m_cacheProvider->processRequestOverview(head()->bbox()); 
      break;

    case RequestEntry::Info:
      qDebug() << __FUNCTION__ << "Pending Info request";
      m_cacheProvider->processRequestInfo(head()->cache()); 
      break;

    default:
      Q_ASSERT(0);
      break;
    }    
  } else
    qDebug() << __FUNCTION__ << "No pending request";
}

void CacheProvider::RequestQueue::restart() {
  if((size() == 1) && m_cacheProvider->initialized())
    next();
}

CacheProvider::CacheProvider() {
  qDebug() << __FUNCTION__;
  this->m_pending = new RequestQueue(this);
}

CacheProvider::~CacheProvider() {
  qDebug() << __FUNCTION__;
  delete this->m_pending;
}

void CacheProvider::requestOverview(const QGeoBoundingBox &area) {
  qDebug() << __PRETTY_FUNCTION__ << "from" << area.topLeft() << 
    " to " << area.bottomRight();

  m_pending->add(RequestEntry::Overview, area);
  m_pending->restart();
}

void CacheProvider::requestInfo(const QString &cache) {
  qDebug() << __PRETTY_FUNCTION__ << cache;

  m_pending->add(RequestEntry::Info, cache);
  m_pending->restart();
}
