/*
 *  Copyright 2010 Ruediger Gad
 *
 *  This file is part of Binary Clock.
 *
 *  Binary Clock is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Binary Clock is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Binary Clock.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <QPainterPath>
#include <QSettings>

#include "binarydisplaywidget.h"
#include "settingsdialog.h"

BinaryDisplayWidget::BinaryDisplayWidget(int bits, int value, int widgetSize, QWidget *parent) :
    QWidget(parent)
{
    this->bits = bits;
    this->value = value;

    if(widgetSize != 0){
        this->widgetSize = widgetSize;
    }else{
        this->widgetSize = bits;
    }

    ledPen = QPen();
    ledPen.setWidth(DEFAULT_LED_PEN_WIDTH);

    applySettings();
}

void BinaryDisplayWidget::applySettings(){
    QSettings settings;

    ledBorder = settings.value(SettingsDialog::LED_BORDER,
                                SettingsDialog::LED_BORDER_DEFAULT).toInt();
    ledColor = settings.value(SettingsDialog::CLOCK_LED_COLOR,
                                SettingsDialog::CLOCK_LED_COLOR_DEFAULT).value<QColor>();
    ledSize = settings.value(SettingsDialog::LED_SIZE,
                                SettingsDialog::LED_SIZE_DEFAULT).toInt();
    useGrayCode = settings.value(SettingsDialog::USE_GRAY_CODE,
                                 SettingsDialog::USE_GRAY_CODE_DEFAULT).toBool();

    emit update();
}

void BinaryDisplayWidget::drawLed(int x, int y, bool on, QPainter *painter){
    int elementSize = getElementSize();

    QRadialGradient gradient(QPointF(x + (elementSize * 0.5),
                                     y + (elementSize * 0.5)),
                             (elementSize * 0.5));
    gradient.setFocalPoint(x + (elementSize * 0.42),
                           y + (elementSize * 0.4));
    gradient.setColorAt(0, QColor(Qt::white));

    if(on){
        gradient.setColorAt(0.2, ledColor);
        gradient.setColorAt(0.55, ledColor);
        gradient.setColorAt(1, QColor(Qt::black));
    }else{
        gradient.setColorAt(0.2, QColor(Qt::black));
    }

    QPainterPath path;
    path.arcTo(x + ledBorder,
               y + ledBorder,
               ledSize,
               ledSize,
               0, 5760);
    painter->fillPath(path, gradient);

    painter->setPen(ledPen);
    painter->drawArc(x + ledBorder,
                     y + ledBorder,
                     ledSize,
                     ledSize,
                     0, 5760);
}

int BinaryDisplayWidget::getElementSize(){
    return (ledSize + (2 * ledBorder));
}

void BinaryDisplayWidget::paintEvent(QPaintEvent */*event*/){
    QPainter painter(this);

    painter.setRenderHint(QPainter::Antialiasing, true);
    painter.setBackgroundMode(Qt::TransparentMode);

    draw(&painter);

    painter.end();
}

void BinaryDisplayWidget::updateValue(int value){
    qDebug("%d", value);
    if(useGrayCode){
        value = value ^ (value >> 1);
    }

    if(this->value != value){
        this->value = value;
        emit update();
    }
}

void BinaryDisplayWidget::setLedBorder(int border){
    if(this->ledBorder != border){
        this->ledBorder = border;
        emit update();
    }
}

void BinaryDisplayWidget::setLedColor(const QColor &color){
    ledColor = color;
    emit update();
}

void BinaryDisplayWidget::setLedPen(const QPen &pen){
    ledPen = pen;
    emit update();
}

void BinaryDisplayWidget::setLedSize(int size){
    if(ledSize != size){
        ledSize = size;
        emit update();
    }
}
