/*
 * position-publisher.c - Source for PositionPublisher
 * Copyright (C) 2010 Guillaume Desmottes
 * @author Guillaume Desmottes <gdesmott@gnome.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <stdio.h>
#include <stdlib.h>

#include <telepathy-glib/dbus.h>
#include <telepathy-glib/interfaces.h>

#include <location/location-gps-device.h>

#include "connection-watcher.h"
#include "position-publisher.h"

G_DEFINE_TYPE(PositionPublisher, position_publisher, G_TYPE_OBJECT)

/* properties */
enum
{
  PROP_BLUR = 1,
  LAST_PROPERTY
};

enum
{
  SIG_HAS_CONNECTIONS_CHANGED,
  LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0 };

/* Minimum time before 2 publishing (in seconds) */
#define PUBLISH_THROTTLE 10

/* private structure */
typedef struct _PositionPublisherPrivate PositionPublisherPrivate;

struct _PositionPublisherPrivate
{
  ConnectionWatcher *watcher;
  LocationGPSDevice *gps_device;
  /* List of (TpConnection *) supporting location publishing */
  GSList *connections;
  GHashTable *location;
  /* If not 0, we are waiting before publishing again */
  guint throttle_timeout;
  /* TRUE if location has been modified while we were waiting */
  gboolean modified;
  gboolean blur;

  gboolean dispose_has_run;
};

#define POSITION_PUBLISHER_GET_PRIVATE(o)     (G_TYPE_INSTANCE_GET_PRIVATE ((o), POSITION_PUBLISHER_TYPE, PositionPublisherPrivate))

static void conn_invalidated_cb (TpProxy *conn,
    guint domain,
    gint code,
    gchar *message,
    PositionPublisher *self);

static void publish_to_all (PositionPublisher *self);

static void
remove_connection (PositionPublisher *self,
    TpConnection *conn)
{
  PositionPublisherPrivate *priv = POSITION_PUBLISHER_GET_PRIVATE (self);

  g_signal_handlers_disconnect_by_func (conn, G_CALLBACK (conn_invalidated_cb),
      self);
  priv->connections = g_slist_remove (priv->connections, conn);
  g_object_unref (conn);

  if (g_slist_length (priv->connections) == 0)
    {
      g_print ("We don't have location connection any more\n");
      g_signal_emit (self, signals[SIG_HAS_CONNECTIONS_CHANGED], 0, FALSE);
    }
}

static void
set_location_cb (TpConnection *conn,
    const GError *error,
    gpointer user_data,
    GObject *weak_object)
{
  PositionPublisher *self = POSITION_PUBLISHER (weak_object);

  if (error != NULL)
    {
      g_print ("SetLocation failed (%s): %s\n", tp_proxy_get_object_path (conn),
          error->message);

      if (error->code == TP_ERROR_NOT_IMPLEMENTED)
        {
          g_print ("remove connection\n");
          remove_connection (self, conn);
        }

      return;
    }

  g_print ("SetLocation succeed (%s)\n", tp_proxy_get_object_path (conn));
}

static gboolean
publish_throttle_timeout_cb (gpointer data)
{
  PositionPublisher *self = data;
  PositionPublisherPrivate *priv = POSITION_PUBLISHER_GET_PRIVATE (self);

  priv->throttle_timeout = 0;

  if (priv->modified)
    {
      publish_to_all (self);
      priv->modified = FALSE;
    }

  return FALSE;
}

static void
publish_to_conn (PositionPublisher *self,
    TpConnection *conn)
{
  PositionPublisherPrivate *priv = POSITION_PUBLISHER_GET_PRIVATE (self);

  if (priv->location == NULL)
    return;

  tp_cli_connection_interface_location_call_set_location (conn, -1,
      priv->location, set_location_cb, NULL, NULL, G_OBJECT (self));
}

static void
publish_to_all (PositionPublisher *self)
{
  PositionPublisherPrivate *priv = POSITION_PUBLISHER_GET_PRIVATE (self);
  GSList *l;

  if (priv->throttle_timeout != 0)
    /* We are waiting */
    return;

  for (l = priv->connections; l != NULL; l = g_slist_next (l))
    {
      TpConnection *conn = l->data;

      publish_to_conn (self, conn);
    }

  /* We won't publish during the next PUBLISH_THROTTLE seconds */
  priv->throttle_timeout = g_timeout_add_seconds (PUBLISH_THROTTLE,
      publish_throttle_timeout_cb, self);
}

static void
update_position (PositionPublisher *self,
    gdouble lat,
    gdouble lon,
    gdouble alt,
    gdouble accuracy)
{
  PositionPublisherPrivate *priv = POSITION_PUBLISHER_GET_PRIVATE (self);

  if (priv->blur)
    {
      /* Truncate at 1 decimal place */
      lon = ((int) (lon * 10)) / 10.0;
      lat = ((int) (lat * 10)) / 10.0;

      /* FIXME: change accuracy (not that easy as accuracy is in meters) */
    }

  g_print ("update position: lat: %f  lon:  %f  alt: %f accuracy: %f\n",
      lat, lon, alt, accuracy);

  if (priv->location != NULL)
    g_hash_table_unref (priv->location);

  priv->location = tp_asv_new (
      "timestamp", G_TYPE_INT64, (gint64) time (NULL),
      "lat", G_TYPE_DOUBLE, lat,
      "lon", G_TYPE_DOUBLE, lon,
      "alt", G_TYPE_DOUBLE, alt,
      "accuracy", G_TYPE_DOUBLE, accuracy,
      NULL);

  priv->modified = TRUE;

  publish_to_all (self);
}

static void
location_changed_cb (LocationGPSDevice *device,
    PositionPublisher *self)
{
  if (device == NULL)
    return;

  if (device->fix == NULL)
    return;

  if (!(device->fix->fields & LOCATION_GPS_DEVICE_LATLONG_SET))
    return;

  update_position (self, device->fix->latitude, device->fix->longitude,
      device->fix->altitude, device->fix->eph / 100.0);
}

static void
conn_invalidated_cb (TpProxy *conn,
    guint domain,
    gint code,
    gchar *message,
    PositionPublisher *self)
{
  g_print ("connection %s invalidated; removing\n", tp_proxy_get_object_path (
        conn));

  remove_connection (self, TP_CONNECTION (conn));
}

static void
connection_added_cb (ConnectionWatcher *watcher,
    TpConnection *conn,
    PositionPublisher *self)
{
  PositionPublisherPrivate *priv = POSITION_PUBLISHER_GET_PRIVATE (self);

  if (g_slist_find (priv->connections, conn) != NULL)
    return;

  if (!tp_proxy_has_interface_by_id (conn,
        TP_IFACE_QUARK_CONNECTION_INTERFACE_LOCATION))
    return;

  priv->connections = g_slist_prepend (priv->connections, g_object_ref (conn));

  if (g_slist_length (priv->connections) == 1)
    {
      /* We just added the first connection */
      g_print ("We have location connection\n");
      g_signal_emit (self, signals[SIG_HAS_CONNECTIONS_CHANGED], 0, TRUE);
    }

  g_signal_connect (conn, "invalidated",
      G_CALLBACK (conn_invalidated_cb), self);

  publish_to_conn (self, conn);
}

static void
position_publisher_init (PositionPublisher *obj)
{
  PositionPublisherPrivate *priv = POSITION_PUBLISHER_GET_PRIVATE (obj);

  priv->watcher = connection_watcher_new ();

  g_signal_connect (priv->watcher, "connection-added",
      G_CALLBACK (connection_added_cb), obj);

  priv->gps_device = g_object_new (LOCATION_TYPE_GPS_DEVICE, NULL);

  g_signal_connect (priv->gps_device, "changed",
      G_CALLBACK (location_changed_cb), obj);

  priv->connections = NULL;
}

static void
position_publisher_constructed (GObject *object)
{
  PositionPublisher *self = POSITION_PUBLISHER (object);
  PositionPublisherPrivate *priv = POSITION_PUBLISHER_GET_PRIVATE (self);

  connection_watcher_start (priv->watcher);

  if (G_OBJECT_CLASS (position_publisher_parent_class)->constructed)
    G_OBJECT_CLASS (position_publisher_parent_class)->constructed (object);
}

static void position_publisher_dispose (GObject *object);
static void position_publisher_finalize (GObject *object);
static void position_publisher_get_property (GObject *object,
    guint property_id, GValue *value, GParamSpec *pspec);
static void position_publisher_set_property (GObject *object,
    guint property_id, const GValue *value, GParamSpec *pspec);

static void
position_publisher_class_init (PositionPublisherClass *position_publisher_class)
{
  GObjectClass *object_class = G_OBJECT_CLASS (position_publisher_class);
  GParamSpec *param_spec;

  g_type_class_add_private (position_publisher_class, sizeof (PositionPublisherPrivate));

  object_class->dispose = position_publisher_dispose;
  object_class->finalize = position_publisher_finalize;
  object_class->get_property = position_publisher_get_property;
  object_class->set_property = position_publisher_set_property;

  object_class->constructed = position_publisher_constructed;

  param_spec = g_param_spec_boolean ("blur", "Blur?",
      "Whether the real GPS position is truncated for privacy concerns.", TRUE,
      G_PARAM_CONSTRUCT | G_PARAM_WRITABLE | G_PARAM_READABLE
      | G_PARAM_STATIC_STRINGS);
  g_object_class_install_property (object_class, PROP_BLUR, param_spec);

  signals[SIG_HAS_CONNECTIONS_CHANGED] = g_signal_new (
      "has-connections-changed",
      G_OBJECT_CLASS_TYPE (object_class),
      G_SIGNAL_RUN_LAST,
      0,
      NULL, NULL,
      g_cclosure_marshal_VOID__BOOLEAN,
      G_TYPE_NONE,
      1, G_TYPE_BOOLEAN);
}

void
position_publisher_dispose (GObject *object)
{
  PositionPublisher *self = POSITION_PUBLISHER (object);
  PositionPublisherPrivate *priv = POSITION_PUBLISHER_GET_PRIVATE (self);
  GSList *l;

  if (priv->dispose_has_run)
    return;

  priv->dispose_has_run = TRUE;

  g_object_unref (priv->watcher);
  g_object_unref (priv->gps_device);

  for (l = priv->connections; l != NULL; l = g_slist_next (l))
    {
      g_object_unref (l->data);
    }

  g_hash_table_unref (priv->location);

  if (priv->throttle_timeout != 0)
    g_source_remove (priv->throttle_timeout);

  if (G_OBJECT_CLASS (position_publisher_parent_class)->dispose)
    G_OBJECT_CLASS (position_publisher_parent_class)->dispose (object);
}

void
position_publisher_finalize (GObject *object)
{
  PositionPublisher *self = POSITION_PUBLISHER (object);
  PositionPublisherPrivate *priv = POSITION_PUBLISHER_GET_PRIVATE (self);

  g_slist_free (priv->connections);

  G_OBJECT_CLASS (position_publisher_parent_class)->finalize (object);
}

static void
position_publisher_get_property (GObject *object,
                                 guint property_id,
                                 GValue *value,
                                 GParamSpec *pspec)
{
  PositionPublisher *self = POSITION_PUBLISHER (object);
  PositionPublisherPrivate *priv = POSITION_PUBLISHER_GET_PRIVATE (self);

  switch (property_id) {
    case PROP_BLUR:
      g_value_set_boolean (value, priv->blur);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
  }
}

static void
position_publisher_set_property (GObject *object,
                                 guint property_id,
                                 const GValue *value,
                                 GParamSpec *pspec)
{
  PositionPublisher *self = POSITION_PUBLISHER (object);
  PositionPublisherPrivate *priv = POSITION_PUBLISHER_GET_PRIVATE (self);

  switch (property_id) {
    case PROP_BLUR:
      priv->blur = g_value_get_boolean (value);
      break;

      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
  }
}

PositionPublisher *
position_publisher_new (void)
{
  return g_object_new (POSITION_PUBLISHER_TYPE,
      NULL);
}

void
position_publisher_set_blur (PositionPublisher *self,
    gboolean blur)
{
  PositionPublisherPrivate *priv = POSITION_PUBLISHER_GET_PRIVATE (self);

  g_print ("%s blurring location\n", blur ? "Start": "Stop");
  priv->blur = blur;
  g_object_notify (G_OBJECT (self), "blur");
}

gboolean
position_publisher_has_connections (PositionPublisher *self)
{
  PositionPublisherPrivate *priv = POSITION_PUBLISHER_GET_PRIVATE (self);

  return g_slist_length (priv->connections) > 0;
}
