/*
 * azimuth.c - Source for Azimith
 * Copyright (C) 2010 Guillaume Desmottes
 * @author Guillaume Desmottes <gdesmott@gnome.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <stdio.h>
#include <stdlib.h>

#include <gconf/gconf-client.h>
#include <telepathy-glib/util.h>

#include <location/location-gpsd-control.h>

#include "azimuth.h"
#include "azimuth-gconf.h"
#include "position-publisher.h"

G_DEFINE_TYPE(Azimuth, azimuth, G_TYPE_OBJECT)

/* private structure */
typedef struct _AzimuthPrivate AzimuthPrivate;

struct _AzimuthPrivate
{
  GMainLoop *loop;
  PositionPublisher *publisher;
  GConfClient *gconf;
  LocationGPSDControl *gps_control;
};

#define AZIMUTH_GET_PRIVATE(o)     (G_TYPE_INSTANCE_GET_PRIVATE ((o), AZIMUTH_TYPE, AzimuthPrivate))

static void
update_gps (Azimuth *self)
{
  AzimuthPrivate *priv = AZIMUTH_GET_PRIVATE (self);
  gboolean start_gps;

  start_gps = gconf_client_get_bool (priv->gconf,
      AZIMUTH_GCONF_KEY_START_GPS, NULL);

  if (priv->publisher != NULL &&
      start_gps &&
      position_publisher_has_connections (priv->publisher))
    {
      g_print ("starting GPS\n");
      location_gpsd_control_start (priv->gps_control);
    }
  else
    {
      g_print ("stopping GPS\n");
      location_gpsd_control_stop (priv->gps_control);
    }
}

static void
update_blur (Azimuth *self)
{
  AzimuthPrivate *priv = AZIMUTH_GET_PRIVATE (self);
  gboolean blur;

  if (priv->publisher == NULL)
    return;

  blur = gconf_client_get_bool (priv->gconf,
      AZIMUTH_GCONF_KEY_BLUR, NULL);

  position_publisher_set_blur (priv->publisher, blur);
}

static void
has_connections_changed_cb (PositionPublisher *publisher,
    gboolean has_connections,
    Azimuth *self)
{
  update_gps (self);
}

static void
create_publisher (Azimuth *self)
{
  AzimuthPrivate *priv = AZIMUTH_GET_PRIVATE (self);

  priv->publisher = position_publisher_new ();

  g_signal_connect (priv->publisher, "has-connections-changed",
      G_CALLBACK (has_connections_changed_cb), self);
}

static void
enabled_changed (Azimuth *self,
    gboolean enabled)
{
  AzimuthPrivate *priv = AZIMUTH_GET_PRIVATE (self);

  if (enabled)
    {
      if (priv->publisher != NULL)
        return;

      g_print ("enable publishing\n");
      create_publisher (self);

      update_blur (self);
    }
  else
    {
      g_print ("disable publishing\n");
      if (priv->publisher == NULL)
        return;

      g_object_unref (priv->publisher);
      priv->publisher = NULL;
    }

  update_gps (self);
}

static void
gconf_notification_cb (GConfClient *client,
    guint cnxn_id,
    GConfEntry *entry,
    gpointer user_data)
{
  Azimuth *self = user_data;
  const gchar *key = gconf_entry_get_key (entry);
  GConfValue *value = gconf_entry_get_value (entry);

  if (!tp_strdiff (key, AZIMUTH_GCONF_KEY_ENABLED) &&
      value->type == GCONF_VALUE_BOOL)
    {
      gboolean enabled = gconf_value_get_bool (value);

      enabled_changed (self, enabled);
    }

  else if (!tp_strdiff (key, AZIMUTH_GCONF_KEY_START_GPS) &&
      value->type == GCONF_VALUE_BOOL)
    {
      update_gps (self);
    }

  else if (!tp_strdiff (key, AZIMUTH_GCONF_KEY_BLUR) &&
      value->type == GCONF_VALUE_BOOL)
    {
      update_blur (self);
    }
}

static void
azimuth_init (Azimuth *self)
{
  AzimuthPrivate *priv = AZIMUTH_GET_PRIVATE (self);

  priv->loop = g_main_loop_new (NULL, FALSE);
  priv->publisher = NULL;

  priv->gconf = gconf_client_get_default ();

  gconf_client_add_dir (priv->gconf, AZIMUTH_GCONF_SECTION,
      GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);

  gconf_client_notify_add (priv->gconf, AZIMUTH_GCONF_SECTION,
      gconf_notification_cb, self, NULL, NULL);

  /* GPS controller */
  priv->gps_control = location_gpsd_control_get_default();

  g_object_set (G_OBJECT(priv->gps_control),
    "preferred-method", LOCATION_METHOD_USER_SELECTED,
    "preferred-interval", LOCATION_INTERVAL_120S,
    NULL);
}

static void
azimuth_dispose (GObject *object)
{
  Azimuth *self = AZIMUTH (object);
  AzimuthPrivate *priv = AZIMUTH_GET_PRIVATE (self);

  if (priv->publisher != NULL)
    {
      g_object_unref (priv->publisher);
      priv->publisher = NULL;
    }

  if (priv->loop != NULL)
    {
      g_main_loop_unref (priv->loop);
      priv->loop = NULL;
    }

  if (priv->gconf != NULL)
    {
      g_object_unref (priv->gconf);
      priv->gconf = NULL;
    }

  if (priv->gps_control != NULL)
    {
      location_gpsd_control_stop (priv->gps_control);
      g_object_unref (priv->gps_control);
      priv->gps_control = NULL;
    }

  if (G_OBJECT_CLASS (azimuth_parent_class)->dispose)
    G_OBJECT_CLASS (azimuth_parent_class)->dispose (object);
}

static void
azimuth_class_init (AzimuthClass *azimuth_class)
{
  GObjectClass *object_class = G_OBJECT_CLASS (azimuth_class);

  g_type_class_add_private (azimuth_class, sizeof (AzimuthPrivate));

  object_class->dispose = azimuth_dispose;
}

Azimuth *
azimuth_new (void)
{
  return g_object_new (AZIMUTH_TYPE,
      NULL);
}

void
azimuth_run (Azimuth *self)
{
  AzimuthPrivate *priv = AZIMUTH_GET_PRIVATE (self);
  gboolean enabled;

  enabled = gconf_client_get_bool (priv->gconf, AZIMUTH_GCONF_KEY_ENABLED,
      NULL);
  if (enabled)
    {
      g_print ("publishing is enabled\n");
      g_assert (priv->publisher == NULL);
      create_publisher (self);

      update_gps (self);
      update_blur (self);
    }
  else
    {
      g_print ("publishing is disabled\n");
    }

  g_print ("azimuth running\n");
  g_main_loop_run (priv->loop);
}
