#!/usr/bin/env python
# -*- coding: utf-8 -*-

import errno
import re
import sys

from subprocess    import call

from PyQt4.QtCore  import Qt, QObject, QString, QStringList, QVariant, \
                          SIGNAL, SLOT
from PyQt4.QtGui   import QAbstractItemView, QComboBox, QDialog, \
                          QDialogButtonBox, QHBoxLayout, QInputDialog, \
                          QLabel, QLineEdit, QListWidget, QMenuBar, \
                          QMessageBox, QPushButton, QStringListModel, \
                          QVBoxLayout, QWidget
from ACE_file      import ACEFile, ACEFileReadException
from ACE_utils     import apply_blacklist
from ACE_settings  import ACESettings
from ACE_encoding  import ACEEncodingDecodeException

version = '0.0.9'

langLookup = {
    3: QString.fromUtf8('English (United Kingdom)'),
    4: QString.fromUtf8('Français (France)'),
    5: QString.fromUtf8('Deutsch (Deutschland)'),
    6: QString.fromUtf8('Español (España)'),
    8: QString.fromUtf8('Português (Portugal)'),
    9: QString.fromUtf8('Svenska (Sverige)'),
    10: QString.fromUtf8('Suomi (Suomi)'),
    11: QString.fromUtf8('Norsk (Norge)'),
    12: QString.fromUtf8('Dansk (Danmark)'),
    13: QString.fromUtf8('Nederlands (Nederland)'),
    14: QString.fromUtf8('Ελληνικά (Ελλάδα)'),
    22: QString.fromUtf8('Polski (Polska)'),
    23: QString.fromUtf8('Čeština (Česká republika)'),
    39: QString.fromUtf8('Русский (Россия)'),
    46: QString.fromUtf8('English (United States of America)'),
    47: QString.fromUtf8('Italiano (Italia)'),
    51: QString.fromUtf8('Español (América Latina)'),
    52: QString.fromUtf8('Français (Québec)'),
}

delModes = QStringList(('Normal', 'Confirm', 'OnClick', 'Blacklist'))
encodings = QStringList(('Default', 'CP1252', 'ISO8859-1', 'ISO8859-2', 'ISO8859-3', 'ISO8859-4', 'ISO8859-5', 'ISO8859-6', 'ISO8859-7', 'ISO8859-8', 'ISO8859-9', 'ISO8859-10', 'ISO8859-11', 'ISO8859-13', 'ISO8859-14', 'ISO8859-15', 'ISO8859-16', 'UTF-8'))

class ACEAboutWindow(QDialog): #{{{1
    def __init__(self, parent): #{{{2
        QDialog.__init__(self, parent)

        self.setWindowTitle('About')

        aboutText  = QString('<h2>Auto-Complete Editor v%1</h2>').arg(version)

        aboutText += "<p>An editor for the user's auto-complete dictionary</p>"
        aboutText += QString('<p>Written by %1<br />').arg('Robin Hill <a href="mailto:maemo@robinhill.me.uk">&lt;maemo@robinhill.me.uk&gt;</a>')
        aboutText += QString('Please email or post to %1 for support</p>').arg('<a href="http://talk.maemo.org/">t.m.o</a>')

        txt = QLabel()
        txt.setOpenExternalLinks(True)
        txt.setTextFormat(Qt.RichText)
        txt.setText(aboutText)

        lyt = QHBoxLayout(self)
        lyt.addWidget(txt)

class ACEConfigWindow(QDialog): #{{{1
    def __init__(self, parent): #{{{2
        QDialog.__init__(self, parent)

        self._parent = parent
        self.setWindowTitle('Settings')

        lyt = QVBoxLayout(self)

        if parent.maemo:
            from PyQt4.QtMaemo5 import QMaemo5ListPickSelector, QMaemo5ValueButton

            lstDelMode = QStringListModel(delModes)
            lstEncoding = QStringListModel(encodings)

            selDelMode = QMaemo5ListPickSelector()
            selDelMode.setModel(lstDelMode)
            selDelMode.setCurrentIndex(delModes.indexOf(parent.delMode))

            selEncoding = QMaemo5ListPickSelector()
            selEncoding.setModel(lstEncoding)
            selEncoding.setCurrentIndex(encodings.indexOf(parent.encOverride))

            btnDelMode = QMaemo5ValueButton('Delete mode')
            btnDelMode.setPickSelector(selDelMode)

            btnEncoding = QMaemo5ValueButton('Character encoding')
            btnEncoding.setPickSelector(selEncoding)

            lyt.addWidget(btnDelMode)
            lyt.addWidget(btnEncoding)

            QObject.connect(selDelMode, SIGNAL('selected(QString)'), self.changeDeleteMode)
            QObject.connect(selEncoding, SIGNAL('selected(QString)'), self.changeEncoding)

            # If these are not saved, we get a SegFault
            self._selDelMode  = selDelMode
            self._selEncoding = selEncoding

            # If these are not saved, the popup lists are empty
            self._lstDelMode  = lstDelMode
            self._lstEncoding = lstEncoding
        else:
            boxLytDelMode = QHBoxLayout()
            boxLytEncoding = QHBoxLayout()

            lblDelMode = QLabel('Delete mode:')
            lblEncoding = QLabel('Character encoding:')

            btnDelMode = QComboBox()
            btnDelMode.addItems(delModes)
            btnDelMode.setCurrentIndex(delModes.indexOf(parent.delMode))

            btnEncoding = QComboBox()
            btnEncoding.addItems(encodings)
            btnEncoding.setCurrentIndex(encodings.indexOf(parent.encOverride))

            boxLytDelMode.addWidget(lblDelMode)
            boxLytDelMode.addWidget(btnDelMode)
            lyt.addLayout(boxLytDelMode)

            boxLytEncoding.addWidget(lblEncoding)
            boxLytEncoding.addWidget(btnEncoding)
            lyt.addLayout(boxLytEncoding)

            QObject.connect(btnDelMode, SIGNAL('currentIndexChanged(int)'), self.changeDeleteModeIndex)
            QObject.connect(btnEncoding, SIGNAL('currentIndexChanged(int)'), self.changeEncodingIndex)

        btns = QDialogButtonBox(QDialogButtonBox.Ok|QDialogButtonBox.Cancel)
        QObject.connect(btns, SIGNAL('accepted()'), self, SLOT('accept()'))
        QObject.connect(btns, SIGNAL('rejected()'), self, SLOT('reject()'))
        lyt.addWidget(btns)

    def changeDeleteModeIndex(self, modeIndex): #{{{2
        mode = delModes[modeIndex]

        self.changeDeleteMode(mode)

    def changeDeleteMode(self, mode): #{{{2
        if mode != self._parent.newDelMode:
            self._parent.newDelMode = mode

    def changeEncodingIndex(self, encIndex): #{{{2
        enc = encodings[encIndex]

        self.changeEncoding(enc)

    def changeEncoding(self, enc): #{{{2
        if enc != self._parent.encOverride:
            self._parent.encOverride = enc

class ACEViewWindow(QWidget): #{{{1
    dicts = {}
    maemo = False
    delMode = 'Normal'

    def __init__(self, parent): #{{{2
        QWidget.__init__(self, parent)

        if self.maemo:
            self.setAttribute(Qt.WA_Maemo5StackedWindow);

    def _applyDeleteMode(self): #{{{2
        if self.newDelMode != self.delMode:
            if self.delMode == 'OnClick':
                self._delBtn.setEnabled(True)
                QObject.disconnect(self._listView, SIGNAL('itemClicked(QListWidgetItem *)'), self.deleteEntry)
            elif self.delMode == 'Blacklist':
                self._delBtn.setText('Delete')

            if self.newDelMode == 'OnClick':
                self._delBtn.setEnabled(False)
                QObject.connect(self._listView, SIGNAL('itemClicked(QListWidgetItem *)'), self.deleteEntry)
            elif self.newDelMode == 'Blacklist':
                self._delBtn.setText('Blacklist')

            self.delMode = self.newDelMode

    def _getLangName(self, langId): #{{{2
        if langId in langLookup:
            return langLookup[langId]
        else:
            QMessageBox.information(self, 'Unknown dictionary', QString('An unknown dictionary ID has been found.\nPlease contact\n\tRobin Hill <maemo@robinhill.me.uk>\nwith:\n\t- the ID: %1\n\t- the dictionary language you have selected (from Settings/Text input)').arg(langId))
            return 'Unknown'

    def addEntry(self, checked): #{{{2
        dlg = QDialog(self)
        dlg.setWindowTitle('Enter new value')

        txtBox = QLineEdit()
        txtBox.setInputMethodHints(Qt.ImhNoAutoUppercase|Qt.ImhNoPredictiveText)

        btnBox = QDialogButtonBox(QDialogButtonBox.Ok|QDialogButtonBox.Cancel)
        QObject.connect(btnBox, SIGNAL('accepted()'), dlg, SLOT('accept()'))
        QObject.connect(btnBox, SIGNAL('rejected()'), dlg, SLOT('reject()'))

        lyt = QHBoxLayout(dlg)
        lyt.addWidget(txtBox)
        lyt.addWidget(btnBox)

        ok = dlg.exec_()
        newText = txtBox.text()

        if not ok or len(newText) == 0:
            return

        lwrText = newText.toLower()

        if lwrText in self.dicts[self._curLang]:
            return

        self._listView.addItem(lwrText)
        self.dicts[self._curLang].append(lwrText)

    def deleteEntries(self, checked): #{{{2
        if self.delMode == 'Confirm':
            if QMessageBox.question(self, 'Delete confirmation', QString('Please confirm that you want to delete the %1 selected entries').arg(len(self._listView.selectedItems())), QMessageBox.Ok|QMessageBox.Cancel) != QMessageBox.Ok:
                return

        for entry in self._listView.selectedItems():
            self.deleteEntry(entry)

    def deleteEntry(self, entry): #{{{2
        row = self._listView.row(entry)
        self._listView.takeItem(row)
        val = entry.data(Qt.DisplayRole).toString()

        self.dicts[self._curLang].removeAt(self.dicts[self._curLang].indexOf(val))

        if self.delMode == 'Blacklist':
            if not self._curLang in self.blacklists:
                self.blacklists[self._curLang] = QStringList()

            self.blacklists[self._curLang].append(val)
        
    def loadDictionary(self, row): #{{{2
        self._listView.clear()

        if row >= 0:
            (dictId, check) = self._dictList.itemData(row).toInt()
            self._curLang = dictId
            self._listView.addItems(self.dicts[dictId])

    def setupScreen(self): #{{{2
        self._dictList = QComboBox()
        self._dictList.setInsertPolicy(QComboBox.NoInsert)
        QObject.connect(self._dictList, SIGNAL('currentIndexChanged(int)'), self.loadDictionary)

        self._listView = QListWidget()
        self._listView.setUniformItemSizes(True)
        self._listView.setSelectionMode(QAbstractItemView.MultiSelection)
        self._listView.setSortingEnabled(True)

        self._btnBox = QDialogButtonBox()
        self._btnBox.setOrientation(Qt.Vertical)
        btn = self._btnBox.addButton('Delete selected', QDialogButtonBox.ActionRole)
        QObject.connect(btn, SIGNAL('clicked(bool)'), self.deleteEntries)
        self._delBtn = btn
        btn = self._btnBox.addButton('Add entry', QDialogButtonBox.ActionRole)
        QObject.connect(btn, SIGNAL('clicked(bool)'), self.addEntry)

        lyt = QVBoxLayout()
        lyt.addWidget(self._dictList)
        lyt.addWidget(self._listView)

        self.layout().addLayout(lyt)
        self.layout().addWidget(self._btnBox)

        self._applyDeleteMode()

class ACEMainWindow(ACEViewWindow): #{{{1
    _dict = None
    blacklists = {}

    def __init__(self): #{{{2
        try:
            from PyQt4.QtMaemo5 import QMaemo5InformationBox
            self.maemo = True
        except:
            pass

        ACEViewWindow.__init__(self, None)

        if self.maemo:
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, True)

        self._blWindow = ACEBlacklistWindow(self)
        self._blWindow.setWindowFlags(self._blWindow.windowFlags() | Qt.Window)
        self._blWindow.setupScreen()

        # Set to the N900 screen size to aid testing
        self.setMinimumSize(800, 480)
        self.setWindowTitle('Auto-Complete Editor')

        self._settings = ACESettings()
        self.newDelMode = self._settings.getDelMode()
        self.encOverride = self._settings.getEncoding()

        QHBoxLayout(self)

    def about(self): #{{{2
        aboutWin = ACEAboutWindow(self)
        aboutWin.exec_()

    def apply_blacklist(self): #{{{2
        for i in self.dicts.keys():
            if i in self.blacklists:
                if apply_blacklist(self.dicts[i], self.blacklists[i]):
                    if i == self._curLang:
                        self._listView.clear()
                        self._listView.addItems(self.dicts[i])

    def edit_blacklist(self): #{{{2
        if not self._curLang in self.blacklists:
            self.blacklists[self._curLang] = QStringList()

        self._blWindow.dicts = self.blacklists
        self._blWindow.loadData(self._curLang)
        self._blWindow.show()

        self.saveBlacklists()

    def config(self): #{{{2
        configWin = ACEConfigWindow(self)
        if configWin.exec_() == QDialog.Rejected:
            self.newDelMode = self.delMode

        if self.newDelMode != self.delMode:
            self._settings.setDelMode(self.newDelMode);

        self._applyDeleteMode()

        if self.encOverride != self._dict.getEncoding():
            try:
                self._dict.setEncoding(self.encOverride)
                self._settings.setEncoding(self.encOverride)
            except ACEEncodingDecodeException, e:
                QMessageBox.critical(self, 'Decode error', 'Unable to decode the dictionary using %s' % self.encOverride)
            finally:
                self.loadData()

    def loadDictionary(self, row): #{{{2
        ACEViewWindow.loadDictionary(self, row)

        if row >= 0:
            self._settings.setDisplayDictionary(self._curLang)

    def clearEntries(self, checked): #{{{2
        if self.delMode == 'Confirm':
            if QMessageBox.question(self, 'Delete confirmation', 'Please confirm that you want to delete all entries', QMessageBox.Ok|QMessageBox.Cancel) != QMessageBox.Ok:
                return

        self._listView.clear()
        self.dicts[self._curLang].clear()

    def revertChanges(self, checked): #{{{2
        self.loadData()

    def saveBlacklists(self): #{{{2
        blacklistLangs = []
        for i in self.blacklists.keys():
            if len(self.blacklists[i]) > 0:
                blacklistLangs.append('%s' % i)

        self._settings.setBlacklist(blacklistLangs, self.blacklists)

    def saveChanges(self, checked): #{{{2
        if self.maemo:
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, True)

        self.saveBlacklists()

        langs = []

        for i in self.dicts.keys():
            if len(self.dicts[i]) > 0:
                langs.append(i)

        for i in self.dicts.keys():
            if len(langs) == 0:
                langs.append(i)
            
            if i in langs:
                self._dict.setDict(i, self.dicts[i])
            else:
                self._dict.deleteDict(i)
            
        self._dict.write()

        call(['/usr/bin/killall', 'hildon-input-method'])

        if self.maemo:
            from PyQt4.QtMaemo5 import QMaemo5InformationBox
            QMaemo5InformationBox.information(self, 'Dictionary saved', QMaemo5InformationBox.DefaultTimeout)
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, False)

        self.loadData()

    def setupMenu(self): #{{{2
        menu = QMenuBar()

        act = menu.addAction('Settings')
        QObject.connect(act, SIGNAL('triggered()'), self.config)
        act = menu.addAction('Edit Blacklist')
        QObject.connect(act, SIGNAL('triggered()'), self.edit_blacklist)
        act = menu.addAction('Apply Blacklist')
        QObject.connect(act, SIGNAL('triggered()'), self.apply_blacklist)
        act = menu.addAction('About')
        QObject.connect(act, SIGNAL('triggered()'), self.about)
        self.layout().setMenuBar(menu)

    def setupScreen(self): #{{{2
        ACEViewWindow.setupScreen(self)

        btn = self._btnBox.addButton('Delete all', QDialogButtonBox.ActionRole)
        QObject.connect(btn, SIGNAL('clicked(bool)'), self.clearEntries)
        btn = self._btnBox.addButton('Save changes', QDialogButtonBox.AcceptRole)
        QObject.connect(btn, SIGNAL('clicked(bool)'), self.saveChanges)
        btn = self._btnBox.addButton('Revert changes', QDialogButtonBox.ResetRole)
        QObject.connect(btn, SIGNAL('clicked(bool)'), self.revertChanges)

        # Clear the progress indicator
        if self.maemo:
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, False)

    def loadData(self): #{{{2
        updateSpinner = False
        if self.maemo and not self.testAttribute(Qt.WA_Maemo5ShowProgressIndicator):
            updateSpinner = True
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, True)

        if self._dict == None:
            self._dict = ACEFile()

            try:
                if self.encOverride != 'Default':
                    self._dict.setEncoding(self.encOverride)
                else:
                    self._dict.read()
            except ACEFileReadException, err:
                langId = err.language
                langName = self._getLangName(langId)

                err_msg  = 'Error reading dictionary\n\n'
                err_msg += '%s for %s' % (err.msg, langName)
                
                if err.entry != None:
                    err_msg += '\nLast entry "%s" removed' % err.entry

                QMessageBox.warning(self, 'Dictionary load error', err_msg)
            except ACEEncodingDecodeException, err:
                QMessageBox.critical(self, 'Decode error', 'Unable to decode the dictionary using %s' % self.encOverride)
            except IOError, err:
                if err.errno == errno.ENOENT:
                    QMessageBox.critical(self, 'Dictionary missing', 'Dictionary file not found')
                    sys.exit()
                else:
                    raise

        curDict = self._settings.getDisplayDictionary()
        dictLangs = self._dict.getLanguages()
        self._dictList.clear()
        self.dicts = {}

        for lang in dictLangs:
            self.dicts[lang] = self._dict.getDict(lang)
            self._dictList.addItem(self._getLangName(lang), QVariant(lang))

        if curDict > 0 and curDict in dictLangs:
            self._curLang = curDict
            self._dictList.setCurrentIndex(dictLangs.index(curDict))
        else:
            if len(dictLangs) > 0:
                self._curLang = dictLangs[0]
                self._dictList.setCurrentIndex(0)

        if len(dictLangs) == 0:
            self._btnBox.setEnabled(False)
        else:
            self._btnBox.setEnabled(True)

        if len(dictLangs) <= 1:
            self._dictList.setEnabled(False)
        else:
            self._dictList.setEnabled(True)
        
        self.blacklists.clear()

        self.blacklists = self._settings.getBlacklist()

        if updateSpinner:
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, False)

class ACEBlacklistWindow(ACEViewWindow): #{{{1
    def __init__(self, parent): #{{{2
        self.maemo   = parent.maemo

        ACEViewWindow.__init__(self, parent)

        self._parent = parent
        self.dicts   = parent.blacklists

        self.setWindowTitle('Blacklist Editor')
        self.newDelMode = 'Normal'

        QHBoxLayout(self)

    def setupScreen(self): #{{{2
        ACEViewWindow.setupScreen(self)

        btn = self._btnBox.addButton('Add dictionary', QDialogButtonBox.ActionRole)
        QObject.connect(btn, SIGNAL('clicked(bool)'), self.addDictionary)
        if len(self.dicts) >= len(self._parent.dicts):
            btn.setEnabled(False)

        btn = self._btnBox.addButton('Delete dictionary', QDialogButtonBox.ActionRole)
        QObject.connect(btn, SIGNAL('clicked(bool)'), self.deleteDictionary)


    def loadData(self, dictId = 0): #{{{2
        updateSpinner = False
        if self.maemo and not self.testAttribute(Qt.WA_Maemo5ShowProgressIndicator):
            updateSpinner = True
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, True)

        dictLangs = self.dicts.keys()
        self._dictList.clear()

        if not dictId in dictLangs:
            dictId = dictLangs[0]

        for lang in dictLangs:
            self._dictList.addItem(self._getLangName(lang), QVariant(lang))

        self._dictList.setCurrentIndex(dictLangs.index(dictId))

        if len(dictLangs) > 1:
            self._dictList.setEnabled(True)
        else:
            self._dictList.setEnabled(False)

        if updateSpinner:
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, False)

    def addDictionary(self, checked): #{{{2
        dlg = QDialog(self)
        dlg.setWindowTitle('Add dictionary')

        cmbBox = QComboBox()
        for lang in self._parent.dicts.keys():
            if not lang in self.dicts:
                cmbBox.addItem(self._getLangName(lang), QVariant(lang))

        btnBox = QDialogButtonBox(QDialogButtonBox.Ok|QDialogButtonBox.Cancel)
        QObject.connect(btnBox, SIGNAL('accepted()'), dlg, SLOT('accept()'))
        QObject.connect(btnBox, SIGNAL('rejected()'), dlg, SLOT('reject()'))

        lyt = QHBoxLayout(dlg)
        lyt.addWidget(cmbBox)
        lyt.addWidget(btnBox)

        ok = dlg.exec_()

        if not ok:
            return

        newDict = cmbBox.itemData(cmbBox.currentIndex()).toInt()[0]
        self.dicts[newDict] = QStringList()
        self.loadData(newDict)

    def deleteDictionary(self, checked): #{{{2
        dictLangs = self.dicts.keys()

        if len(dictLangs) > 1:
            del self.dicts[self._curLang]
        else:
            self.dicts[self._curLang].clear()

        self.loadData()
