'''
Advanced Power Abstract Wrappers
2008-2012(c) Kirill Plyashkevich <ru.spirit@gmail.com>
'''

import dbus
from advpowcommon.util.device import device
from advpowcommon.util.system import system

class AbstractWrapper():

  def __init__(self, bus, check = False):
    self.__bus = bus
    self.__available = False
    self.__signals = {'changed': set()}
    self.__type = self.get_type()
    self.__name = self.get_friendly_name()

    self.__properties = {}
    props = self.get_default_properties()
    for prop in props:
      self.__properties['%s.%s' % (self.__type, prop)] = props[prop]

    self.__dbus_signals = {}
    self.interfaces = {}
    self.interfaces_map = {}

    self.depends = {'os': '', 'packages': []}
    self.pre_init()
    self.__check_available()
    if not check:
      self.__end_init()

  def finish_init(self):
    self.__end_init()

  def __end_init(self):
    if self.__available:
      self.__init_interfaces()
      self.__dbus_signals = self.get_default_dbus_signals()
      self.__init_dbus_signals()
      self.post_init()
      self.get_status()

  def pre_init(self):
    raise NotImplementedError("AbstractWrapper.pre_init() must be implemented")

  def post_init(self):
    raise NotImplementedError("AbstractWrapper.post_init() must be implemented")

  def get_status(self):
    raise NotImplementedError("AbstractWrapper.get_status() must be implemented")

  def connect(self, signal, cb):
    try:
      self.__signals[signal].add(cb)
    except KeyError, e:
      print e

  def get_properties(self):
    return self.__properties

  def get_property(self, name):
    try:
      return self.__properties['%s.%s' % (self.__type, name)]
    except KeyError:
      return None

  def get_default_properties(self):
    raise NotImplementedError("AbstractWrapper.get_default_properties() must be implemented")

  def changed(self, values):
    items = values.iteritems()
    result = {}
    for name, value in items:
      pname = '%s.%s' % (self.__type, name)
      if pname in self.__properties:
        self.__properties[pname] = value
        result[pname] = value
    callbacks = self.__signals['changed']
    for cb in callbacks:
      cb(result)

  def _cleanup(self):
    self.__destroy_dbus_signals()
    self.__destroy_interfaces()
    keys = self.__signals.keys()
    for key in keys:
      self.__signals[key].clear()

  def get_dep_packages(self):
    return self.dep_packages

  def get_name(self):
    return self.__name

  def get_friendly_name(self):
    raise NotImplementedError("AbstractWrapper.get_friendly_name() must be implemented")

  def get_signals(self):
    return self.__signals

  def get_default_dbus_signals(self):
    raise NotImplementedError("AbstractWrapper.get_default_dbus_signals() must be implemented")

  def is_available(self):
    return self.__available

  def __check_available(self):
    self.__available = True
    if self.depends['os']:
      if self.depends['os'] != device.get_property('OSSO_PRODUCT_RELEASE_VERSION'):
        self.__available = False
        return self.__available
    for dp in self.depends['packages']:
      if not system.is_package_version_valid(dp, self.depends['packages'][dp]):
        self.__available = False
        return self.__available
    self.__available = self.__available and self.custom_available()
    return self.__available

  def custom_available(self):
    return True

  def __init_interfaces(self):
    for x in self.interfaces_map:
      self.__init_interface(x)

  def __init_interface(self, x):
    try:
      self.interfaces[x] = dbus.Interface(self.__bus.get_object(self.interfaces_map[x][0], self.interfaces_map[x][1], introspect=False, follow_name_owner_changes=True), self.interfaces_map[x][2])
    except:
      self.interfaces[x] = None

  def __destroy_interfaces(self):
    for key in self.interfaces:
      self.__destroy_interface(key)

  def __destroy_interface(self, x):
    self.interfaces[x] = None

  def __init_dbus_signals(self):
    for key in self.__dbus_signals:
      self.__init_dbus_signal(key)

  def add_dbus_signals(self, signals):
    for key in signals:
      self.__destroy_dbus_signal(key)
    self.__dbus_signals.update(signals)
    for key in signals:
      self.__init_dbus_signal(key)

  def __init_dbus_signal(self, key):
    self.__destroy_dbus_signal(key)
    try:
      self.__dbus_signals[key]['match'] = self.__bus.add_signal_receiver(
        getattr(self, 'handle_%s_%s' % (self.__type, key)),
        **self.__dbus_signals[key])
    except Exception, e:
      print e

  def __destroy_dbus_signals(self):
    for key in self.__dbus_signals:
      self.__destroy_dbus_signal(key)

  def __destroy_dbus_signal(self, key):
    if 'match' in self.__dbus_signals[key]:
      receiver_match = self.__dbus_signals[key].pop('match')
      self.__bus.remove_signal_receiver(receiver_match,**self.__dbus_signals[key])
      receiver_match.remove()
