
'''
Advanced Power Monitor
2008-2012(c) Kirill Plyashkevich <ru.spirit@gmail.com>
Connects to bunch of dbus signals, monitors values change and emmits signal in case of update.
'''

from __future__ import with_statement
import threading
import random
import gobject
import os.path
import time
import dbus, dbus.service
from dbus.mainloop.glib import DBusGMainLoop
from advpowcommon.util.loader import ClassLoader
from advpowcommon.util.device import device
from advpowcommon.util.execn import *
from advpowcommon.util.common import *

class apm(dbus.service.Object):

  appname = 'Advanced Power Monitor'
  version = '0.6.0-6'
  iface = "ru.spirit.advanced_power_monitor"
  opath = "/ru/spirit/advanced_power_monitor"
  wrappers_root = 'wrappers'
  modules_root = 'modules'

  def __init__(self):
    print '%s %s' % (self.appname, self.version)
    print '%s\n' % (device.get_property('OSSO_VERSION'))
    self.__lock = threading.Lock()
    self.__bus = dbus.SystemBus(mainloop = DBusGMainLoop(), private = True)
    dbus.service.Object.__init__(self, dbus.service.BusName(self.iface, self.__bus), self.opath)

    self.__classname = '%s.%s' % (__name__, self.__class__.__name__)
    self.__wrappers = {}
    self.__interfaces = {}
    self.__signals = {}
    self.__signal_queue = []
    self.__min_queue = 1
    self.__clients = set()
    self.__clients_range = [0, 0xFFFFFFFF]
    self.__working = False
    self.__ifaces_map = {}#{'CPUFreq':('org.freedesktop.Hal', '/org/freedesktop/Hal/devices/computer', 'org.freedesktop.Hal.Device.CPUFreq')}
    self.__info_map = {}

    modules = ClassLoader.get_modules_list(os.path.dirname(__file__), self.modules_root)
    del modules[self.modules_root]
    output_loaded = ''
    loaded_counter = 0
    for k,v in modules.iteritems():
      ClassLoader.import_module(k)
      modules_list = ClassLoader.import_modules(v)
      class_list = []
      for m in modules_list:
        module_class = ClassLoader.get_module_class(m)
        if module_class:
          class_list.append(module_class)
      for ref_module in class_list:
        c_module = ref_module()
        if c_module.is_available():
          output_loaded = '%s %s' % (output_loaded, c_module.get_name())
          item = {'value': c_module.value, 'time': c_module.interval,
                  'object': c_module, 'async': c_module.async,
                  'clients': set()}
          self.__info_map[k.replace('%s.' % (self.modules_root), '')] = item
          loaded_counter += 1
          break
        else:
          del c_module
    print 'Loaded modules (%s/%s):%s\n' % (loaded_counter, len(modules), output_loaded)

    print 'Loading wrappers...'
    modules = ClassLoader.get_modules_list(os.path.dirname(__file__), self.wrappers_root)
    del modules[self.wrappers_root]
    for k,v in modules.iteritems():
      ClassLoader.import_module(k)
      modules_list = ClassLoader.import_modules(v)
      class_list = []
      for m in modules_list:
        module_class = ClassLoader.get_module_class(m)
        if module_class:
          class_list.append(module_class)
        else:
          del module_class
      if class_list:
        class_list.reverse()
        self.__wrappers[k.replace('%s.' % (self.wrappers_root), '')] = {'name': None, 'object': None, 'available': {}, 'list': class_list}
    self.__init_wrappers()

    self.__init_interfaces()
#    signal.signal(signal.SIGTERM, self.__cleanup)
    self.__working = True
    self.update_info()
    print '\nLOADED'

  def _cleanup(self):
    self.__cleanup()

  def __cleanup(self, *args):
    with self.__lock:
      active_wrappers = self.__wrappers.values()
      for wr in active_wrappers:
        try:
          if wr['object']:
            wr['object']._cleanup()
        except KeyError:
          pass
      info_values = self.__info_map.values()
      for val in info_values:
        try:
          gobject.source_remove(val['src_evt'])
        except KeyError:
          pass
      self.__destroy_interfaces()
      self.__destroy_signals()
      self.__working = False
    print 'EXIT'
#    sys.exit(0)

#Signals subsystem
  def __init_signals(self):
    for x in self.__signals:
      self.__add_signal(x)

  def __add_signal(self, signal):
    self.__remove_signal(signal)
    try:
      signal_obj = self.__signals[signal]
      signal_obj['obj'] = self.__bus.add_signal_receiver(
        getattr(self, 'handle_%s' % signal),
        signal_name=signal_obj['signal_name'],
        dbus_interface=signal_obj['dbus_interface'],
        path=signal_obj['path'])
    except Exception, e:
      self.__print_exception('__add_signal', str(e))

  def __destroy_signals(self):
    for key in self.__signals:
      self.__remove_signal(key)

  def __remove_signal(self, key):
    try:
      c_signal = self.__signals[key].pop('obj')
      self.__bus.remove_signal_receiver(c_signal, signal_name=self.__signals[key]['signal_name'], dbus_interface=self.__signals[key]['dbus_interface'], path=self.__signals[key]['path'])
      c_signal.remove()
    except KeyError:
      pass
    except Exception, e:
      self.__print_exception('__remove_signal(%s)' % (key), str(e))
#end

#Interfaces subsystem
  def __init_interfaces(self):
    for x in self.__ifaces_map:
      try:
        self.__interfaces[x] = dbus.Interface(self.__bus.get_object(self.__ifaces_map[x][0], self.__ifaces_map[x][1], introspect=False, follow_name_owner_changes=True), self.__ifaces_map[x][2])
      except Exception, e:
        self.__interfaces[x] = None
        self.__print_exception('__init_interfaces', str(e))

  def __destroy_interfaces(self):
    for key in self.__interfaces: self.__interfaces[key] = None
#end

#Wrappers routine

  def __init_wrappers(self):
    for key in self.__wrappers:
      print '\nLOADING %s' % (key)
      self.__check_wrapper(key)
      self.change_wrapper(key, self.get_available_wrappers(key)[0])
      print '[U] %s\n' % str(self.__wrappers[key]['name'])

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', out_signature='as')
  def get_wrappers_types(self):
    return self.__wrappers.keys()

  def __init_wrapper(self, key):
    signals = self.__wrappers[key]['object'].get_signals()
    for signal in signals:
      try:
        self.__wrappers[key]['object'].connect(signal, getattr(self, 'handle_%s' % signal))
      except AttributeError, e:
        self.__print_exception('__init_wrapper', '%s: %s' % (key, str(e)))

  def __check_wrapper(self, key):
    if self.__wrappers[key]['available']:
      self.__wrappers[key]['available'].clear()
    wr_list = enumerate(self.__wrappers[key]['list'])
    for ind, wr in wr_list:
      wrapper = wr(self.__bus, True)
      name = wrapper.get_name()
      if wrapper.is_available():
        self.__wrappers[key]['available'][name] = ind
        print '[L] %s' % (name)
        ind += 1
      else:
        print '[F] %s' % (name)
      wrapper._cleanup()
      del wrapper

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', in_signature='ss')
  def change_wrapper(self, key, new_wrapper):
    with self.__lock:
      wrapper_object = None
      try:
        wrapper_object = self.__wrappers[key]['list'][self.__wrappers[key]['available'][new_wrapper]](self.__bus, True)
      except KeyError:
        return
      new_props = wrapper_object.get_properties()
      clients = dict.fromkeys(new_props.keys(), set())
      if new_wrapper != self.__wrappers[key]['name'] and wrapper_object.is_available():#new_wrapper in self.__wrappers[key]['available'].keys()
        if self.__wrappers[key]['object']:
          props = self.__wrappers[key]['object'].get_properties()
          for pkey in props:
            if pkey in new_props:
              clients[pkey] = self.__info_map[pkey]['clients']
            del self.__info_map[pkey]
          self.__wrappers[key]['object']._cleanup()
        self.__wrappers[key]['name'] = new_wrapper
        self.__wrappers[key]['object'] = wrapper_object
        for pkey in new_props:
          self.__info_map[pkey] = {'value': new_props[pkey], 'async': True, 'clients': clients[pkey]}
        self.__init_wrapper(key)
        self.__wrappers[key]['object'].finish_init()

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', in_signature='s', out_signature='as')
  def get_available_wrappers(self, key):
    result = self.__wrappers[key]['available'].keys()
    #result.reverse()
    if result:
      if result[0] == 'Dummy':
        result.append(result.pop(0))
    else:
      result.append(str(self.get_current_wrapper(key)))
    return result

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', in_signature='s', out_signature='s')
  def get_current_wrapper(self, key):
    return str(self.__wrappers[key]['name'])

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', in_signature='s', out_signature='as')
  def get_current_wrapper_properties(self, key):
    return self.__wrappers[key]['object'].get_properties()

#Client routine
  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', out_signature='x')
  def register(self):
    cl_min = self.__clients_range[0]
    cl_max = self.__clients_range[1]
    while True:
      client = random.randint(cl_min, cl_max)
      if not client in self.__clients:
        self.__clients.add(client)
        return client

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request')
  def unregister(self, client):
    for key in self.__info_map:
      self.__info_map[key]['clients'].discard(client)
    self.__clients.discard(client)

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', in_signature='xas')
  def subscribe(self, client, values=[]):
    client = long(client)
    if client in self.__clients:
      for v in values:
        try:
          self.__info_map[v]['clients'].add(client)
          c_key = self.__info_map[v]
          if len(c_key['clients']) == 1 and 'object' in c_key:
            self.__update_module(v)
          time.sleep(0.1)
        except KeyError:
          pass
      self.__update_queue_size()

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', in_signature='xas')
  def unsubscribe(self, client, values=[]):
    client = long(client)
    if client in self.__clients:
      for v in values:
        try:
          self.__info_map[v]['clients'].discard(client)
          if not self.__info_map[v]['clients']:
            gobject.source_remove(self.__info_map[v]['src_evt'])
        except KeyError:
          pass
      self.__update_queue_size()

  def __update_queue_size(self):
    info_map_values = self.__info_map.values()
    info_map_async = [v for v in info_map_values if not v['async'] and len(v['clients'])]
    total = len(info_map_async)
    watching = sum([len(v['clients']) for v in info_map_async])
    if not total:
      total = 1
      watching = 1
    self.__min_queue = min(watching, 1 + watching/(total*len(self.__clients)))

#Update information
  #def handle_property_changed(self, name, value):
    #self.__check_property_modified([name, value])#, True)

  def handle_changed(self, properties):
    result = properties.items()
    result.append(True)
    self.__check_property_modified(*result)

  def update_info(self):
    for x in self.__info_map:
      if not 'clients' in self.__info_map[x]:
        self.__info_map[x]['clients'] = set()
      if 'object' in self.__info_map[x]:
        self.__update_module(x)
    return self.__working

  def __update_module(self, key):
    try:
      self.__check_property_modified([key, self.__info_map[key]['object'].get_updated()], self.__info_map[key]['object'].async)
    except Exception, e:
      self.__print_exception('__update_module(%s)' % (key), str(e))
    self.__add_timer_event(key)
    return False

#Changes watchers
  @dbus.service.signal(dbus_interface='ru.spirit.advanced_power_monitor.signal', signature='ias')
  def PropertyModified(self, size, keys):
    return size, keys

  def __check_property_modified(self, *values):
    signals = []
    no_queue = False
    if values:
      values = list(values)
      if type(values[-1]) == bool:
        no_queue = values.pop(-1)
      for x in values:
        if self.get_value(x[0]) != x[1]:
          #print x[0], ':' , self.get_value(x[0]), ' -> ', x[1]
          self.__set_value(x[0], x[1])
          if self.__info_map[x[0]]['clients']:
            signals.append(x[0])
      if signals:
        if no_queue:
          self.PropertyModified(len(signals), signals)
        else:
          self.__notify(signals)

  def __notify(self, keys):
    watching = []
    for k in keys:
      try:
        clients_len = len(self.__info_map[k]['clients'])
        if clients_len:
          self.__signal_queue.append(k)
          watching.append(clients_len)
      except KeyError:
        pass
    info_map = self.__info_map.values()
    total = len([v for v in info_map if not v['async'] and v['clients']])
    if not total:
      total = 1
      watching = [1]
    c_queue = min(watching, 1 + sum(watching)/(total*len(self.__clients)))
    if c_queue > self.__min_queue:
      self.__min_queue = c_queue
    if len(self.__signal_queue) >= self.__min_queue:
      result = set(self.__signal_queue)
      del self.__signal_queue[:]
      self.PropertyModified(len(result), result)
#end

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', out_signature='s')
  def get_version(self):
    return self.version

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', out_signature='as')
  def get_property_list(self):
    return self.__info_map.keys()

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', out_signature='v')
  def get_value(self, key):
    try:
      return self.__info_map[str(key)]['value']
    except KeyError:
      return 'Error'

  def __set_value(self, key, value=None):
    try:
      self.__info_map[str(key)]['value'] = value
    except:
      pass

  def __add_timer_event(self, key):
    if self.__working and 'time' in self.__info_map[key] and self.__info_map[key]['time'] > 0 and len(self.__info_map[key]['clients']) > 0 and not self.get_value('system_inactive'):
      self.__info_map[key]['src_evt'] = gobject.timeout_add(self.__info_map[key]['time'], self.__update_module, key)

  def __print_exception(self, func_name, message):
    print_exception_message('%s.%s()' % (self.__classname, func_name), message)
