
'''
Advanced Power Monitor
2008-2010(c) Kirill Plyashkevich <ru.spirit@gmail.com>
Connects to bunch of dbus signals, monitors values change and emmits signal in case of update.
'''

import random, gobject, os.path, signal, sys, time
import dbus, dbus.service
from dbus.mainloop.glib import DBusGMainLoop
from advpowcommon.util.loader import ClassLoader
from advpowcommon.util.device import device

class apm(dbus.service.Object):

  appname = 'Advanced Power Monitor'
  version = '0.5.2-10'
  iface = "ru.spirit.advanced_power_monitor"
  opath = "/ru/spirit/advanced_power_monitor"
  wrappers_root = 'wrappers'
  modules_root = 'modules'

  def __init__(self):
    print '%s %s' % (self.appname, self.version)
    print '%s\n' % (device.get_property('OSSO_VERSION'))
    self.__bus = dbus.SystemBus(mainloop = DBusGMainLoop(), private = True)
    dbus.service.Object.__init__(self, dbus.service.BusName(self.iface, self.__bus), self.opath)

    self.__classname = '%s.%s' % (__name__, self.__class__.__name__)
    self.battery_is_charging= False
    self.battery_is_full = False
    self.battery_charger_is_disconnected = False
    self.__wrappers = {}
    self.__interfaces = {}
    self.__signals = {}
    self.__signal_queue = []
    self.__min_queue = 1
    self.__clients = set()
    self.__clients_range = [0, 0xFFFFFFFF]
    self.__working = False
    self.__ifaces_map = {'CPUFreq':('org.freedesktop.Hal', '/org/freedesktop/Hal/devices/computer', 'org.freedesktop.Hal.Device.CPUFreq')}
    self.__info_map = {'battery_status': {'value': 0, 'async': True, 'clients': set()},
                       'timeleft_idle': {'value': 0, 'async': True, 'clients': set()},
                       'timeleft_active': {'value': 0, 'async': True, 'clients': set()},
                       'device_mode': {'value': 'unknown', 'async': True, 'clients': set()},
                       'system_inactive': {'value': False, 'async': True, 'clients': set()},
                       'bluetooth_mode': {'value': 'unknown', 'async': True, 'clients': set()},
                       'bluetooth_modes': {'value': ['unknown'], 'async': True, 'clients': set()},
                       'usb_mode': {'value': 'unknown', 'async': True, 'clients': set()},
                       'usb_modes': {'value': ['unknown'], 'async': True, 'clients': set()}
                      }

    modules = ClassLoader.get_modules_list(os.path.dirname(__file__), self.modules_root)
    del modules[self.modules_root]
    output_loaded = ''
    loaded_counter = 0
    for k,v in modules.iteritems():
      ClassLoader.import_module(k)
      modules_list = ClassLoader.import_modules(v)
      class_list = []
      for m in modules_list:
        module_class = ClassLoader.get_module_class(m)
        if module_class: class_list.append(module_class)
      if class_list:
        for ref_module in class_list:
          c_module = ref_module()
          if c_module.is_available():
            output_loaded = '%s %s' % (output_loaded, c_module.get_name())
            item = {'value': c_module.value, 'time': c_module.interval, 'object': c_module, 'async': c_module.async, 'clients': set()}
            self.__info_map[k.replace('%s.' % (self.modules_root), '')] = item
            loaded_counter += 1
            break
          else:
            del c_module
    print 'Loaded modules (%s/%s):%s\n' % (loaded_counter, len(modules), output_loaded)

    print 'Loading wrappers...'
    modules = ClassLoader.get_modules_list(os.path.dirname(__file__), self.wrappers_root)
    del modules[self.wrappers_root]
    for k,v in modules.iteritems():
      ClassLoader.import_module(k)
      modules_list = ClassLoader.import_modules(v)
      class_list = []
      for m in modules_list:
        module_class = ClassLoader.get_module_class(m)
        if module_class: class_list.append(module_class)
        else: del module_class
      if class_list: self.__wrappers[k.replace('%s.' % (self.wrappers_root), '')] = {'name': None, 'object': None, 'available': {}, 'list': class_list}
    self.__init_wrappers()

    self.__init_interfaces()
    signal.signal(signal.SIGTERM, self.__cleanup)
    self.__working = True
    self.update_info()
    print '\nLOADED'

  def _cleanup(self):
    self.__cleanup()

  def __cleanup(self, *args):
    active_wrappers = self.__wrappers.values()
    for wr in active_wrappers:
      try:
        if wr['object']: wr['object']._cleanup()
      except KeyError:
        pass
    for x in self.__info_map:
      try: gobject.source_remove(self.__info_map[x]['src_evt'])
      except KeyError: pass
    self.__destroy_interfaces()
    self.__destroy_signals()
    self.__working = False
    print 'EXIT'
    sys.exit(0)

#Signals subsystem
  def __init_signals(self):
    for x in self.__signals: self.__add_signal(x)

  def __add_signal(self, signal):
    self.__remove_signal(signal)
    try: self.__signals[signal]['obj'] = self.__bus.add_signal_receiver(getattr(self, 'handle_%s' % signal), signal_name=self.__signals[signal]['signal_name'], dbus_interface=self.__signals[signal]['dbus_interface'], path=self.__signals[signal]['path'])
    except Exception, e: self.__print_exception('__add_signal', str(e))

  def __destroy_signals(self):
    for key in self.__signals: self.__remove_signal(key)

  def __remove_signal(self, key):
    try:
      c_signal = self.__signals[key]['obj']
      self.__bus.remove_signal_receiver(c_signal)
      self.__bus.clean_up_signal_match(c_signal)
      c_signal.remove()
      self.__signals[key].pop('obj')
    except KeyError:
      pass
    except Exception, e:
      self.__print_exception('__remove_signal(%s)' % (key), str(e))
#end

#Interfaces subsystem
  def __init_interfaces(self):
    for x in self.__ifaces_map:
      try:
        self.__interfaces[x] = dbus.Interface(self.__bus.get_object(self.__ifaces_map[x][0], self.__ifaces_map[x][1]), self.__ifaces_map[x][2])
      except Exception, e:
        self.__interfaces[x] = None
        self.__print_exception('__init_interfaces', str(e))

  def __destroy_interfaces(self):
    for key in self.__interfaces: self.__interfaces[key] = None
#end

#Wrappers routine
  def __init_wrappers(self):
    for key in self.__wrappers:
      print '\nLOADING %s' % (key)
      self.__check_wrapper(key)
      self.change_wrapper(key, self.get_available_wrappers(key)[0])
      print '[U] %s\n' % str(self.__wrappers[key]['name'])

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', out_signature='as')
  def get_wrappers_types(self):
    return self.__wrappers.keys()

  def __init_wrapper(self, key):
    signals = self.__wrappers[key]['object'].get_signals()
    self.__signals.update(signals)
    for s in signals: self.__add_signal(s)

  def __check_wrapper(self, key):
    if self.__wrappers[key]['available']: self.__wrappers[key]['available'].clear()
    wr_list = enumerate(self.__wrappers[key]['list'])
    for ind, wr in wr_list:
      wrapper = wr(self.__bus, True)
      name = wrapper.get_name()
      if wrapper.is_available():
        self.__wrappers[key]['available'][name] = ind
        print '[L] %s' % (name)
        ind += 1
      else:
        print '[F] %s' % (name)
      wrapper._cleanup()
      del wrapper

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', in_signature='ss')
  def change_wrapper(self, key, new_wrapper):
    wrapper_object = None
    try: wrapper_object = self.__wrappers[key]['list'][self.__wrappers[key]['available'][new_wrapper]](self.__bus, True)
    except KeyError: return
    if new_wrapper != self.__wrappers[key]['name'] and wrapper_object.is_available():#new_wrapper in self.__wrappers[key]['available'].keys()
      if self.__wrappers[key]['object']:
        old_wrapper_signals = self.__wrappers[key]['object'].get_signals().keys()
        for sig_key in old_wrapper_signals: self.__remove_signal(sig_key)
        self.__wrappers[key]['object']._cleanup()
      self.__wrappers[key]['name'] = new_wrapper
      wrapper_object._cleanup()#end_init()

      self.__wrappers[key]['object'] = self.__wrappers[key]['list'][self.__wrappers[key]['available'][new_wrapper]](self.__bus)
      self.__init_wrapper(key)
      if key == 'battery':
        self.__update_percent()
        self.__update_timeleft()
      elif key == 'bluetooth':
        try:
          self.__set_value('bluetooth_mode', self.__wrappers[key]['object'].get_mode())
          self.__set_value('bluetooth_modes', self.__wrappers[key]['object'].get_available_modes())
        except Exception, e:
          print e
      elif key == 'device_mode':
        try:
          self.__set_value('device_mode', self.__wrappers[key]['object'].get_device_mode())
          self.__set_value('system_inactive', self.__wrappers[key]['object'].get_system_inactive())
        except Exception, e:
          print e
      elif key == 'usb_mode':
        try:
          self.__set_value('usb_mode', self.__wrappers[key]['object'].get_mode())
          self.__set_value('usb_modes', self.__wrappers[key]['object'].get_available_modes())
        except Exception, e:
          print e

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', in_signature='s', out_signature='as')
  def get_available_wrappers(self, key):
    result = self.__wrappers[key]['available'].keys()
    if result:
      if result[0] == 'Dummy': result.append(result.pop(0))
    else:
      result.append(str(self.get_current_wrapper(key)))
    return result

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', in_signature='s', out_signature='s')
  def get_current_wrapper(self, key):
    return str(self.__wrappers[key]['name'])

#Client routine
  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', out_signature='x')
  def register(self):
    cl_min = self.__clients_range[0]
    cl_max = self.__clients_range[1]
    while True:
      client = random.randint(cl_min, cl_max)
      if not client in self.__clients:
        self.__clients.add(client)
        return client

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request')
  def unregister(self, client):
    for key in self.__info_map: self.__info_map[key]['clients'].discard(client)
    self.__clients.discard(client)

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', in_signature='xas')
  def subscribe(self, client, values=[]):
    client = long(client)
    if client in self.__clients:
      for v in values:
        try:
          self.__info_map[v]['clients'].add(client)
          c_key = self.__info_map[v]
          if len(c_key['clients']) == 1 and 'object' in c_key: self.__update_module(v)
          time.sleep(0.1)
        except KeyError:
          pass
      self.__update_queue_size()

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', in_signature='xas')
  def unsubscribe(self, client, values=[]):
    client = long(client)
    if client in self.__clients:
      for v in values:
        try:
          self.__info_map[v]['clients'].discard(client)
          if not self.__info_map[v]['clients']: gobject.source_remove(self.__info_map[v]['src_evt'])
        except KeyError:
          pass
      self.__update_queue_size()

  def __update_queue_size(self):
    info_map_values = self.__info_map.values()
    info_map_async = [v for v in info_map_values if not v['async'] and len(v['clients'])]
    total = len(info_map_async)
    watching = sum([len(v['clients']) for v in info_map_async])
    if not total:
      total = 1
      watching = 1
    self.__min_queue = min(watching, 1 + watching/(total*len(self.__clients)))

#Update information
  def update_info(self):
    for x in self.__info_map:
      if not 'clients' in self.__info_map[x]: self.__info_map[x]['clients'] = set()
      if 'object' in self.__info_map[x]: self.__update_module(x)
    return self.__working

  def __update_module(self, key):
    try: self.__check_property_modified([key, self.__info_map[key]['object'].get_updated()], self.__info_map[key]['object'].async)
    except Exception, e: self.__print_exception('__update_module(%s)' % (key), str(e))
    self.__add_timer_event(key)
    return False

  def __update_percent(self):
    self.__check_property_modified(['battery_status', self.__wrappers['battery']['object'].update_percent()], True)
    return False

  def __update_timeleft(self):
    timeleft_info = self.__wrappers['battery']['object'].update_timeleft()
    self.__check_property_modified(['timeleft_idle', timeleft_info[0]], ['timeleft_active', timeleft_info[1]], True)
    return False

#Battery signals handlers
  def handle_battery_percentage(self, *args):
    self.__check_property_modified(['battery_status', self.__wrappers['battery']['object'].get_percent(*args)], True)

  def handle_battery_timeleft_info(self, *args):
    timeleft_info = self.__wrappers['battery']['object'].get_timeleft(*args)
    self.__check_property_modified(['timeleft_idle', timeleft_info[0]], ['timeleft_active', timeleft_info[1]])

  def handle_battery_charger_is_connected(self):
    if not self.battery_is_charging:
      self.battery_is_charging= True
      self.battery_charger_is_disconnected = False
      self.PropertyModified(['battery_status'])

  def handle_battery_charger_is_disconnected(self):
    if not self.battery_charger_is_disconnected:
      self.battery_is_full = False
      self.battery_is_charging= False
      self.battery_charger_is_disconnected = True
      self.PropertyModified(['battery_status'])
      self.__set_value('battery_status', 0)
      self.__set_value('timeleft_idle', 0)
      self.__set_value('timeleft_active', 0)
    self.__update_percent()
    self.__update_timeleft()

  def handle_battery_is_full(self):
    if not self.battery_is_full:
      self.battery_is_full = True
      self.PropertyModified(['battery_status'])

#Device mode signal handler
  def handle_device_mode(self, *args):
    self.__wrappers['device_mode']['object'].device_mode_handler(*args)
    self.__check_property_modified(['device_mode', self.__wrappers['device_mode']['object'].get_device_mode()], True)

#System inactive signal handler
  def handle_system_inactive(self, *args):
    self.__wrappers['device_mode']['object'].system_inactive_handler(*args)
    self.__set_value('system_inactive', self.__wrappers['device_mode']['object'].get_system_inactive())
    watching = [k for (k,v) in  self.__info_map.iteritems() if len(v['clients']) and 'object' in v]
    if self.get_value('system_inactive'):
      for key in watching:
        try: gobject.source_remove(self.__info_map[key]['src_evt'])
        except KeyError: pass
    else:
      for key in watching: self.__update_module(key)

#Bluetooth mode signal handler
  def handle_bluetooth_mode(self, *args):
    self.__wrappers['bluetooth']['object'].mode_handler(*args)
    self.__check_property_modified(['bluetooth_mode', self.__wrappers['bluetooth']['object'].get_mode()], True)

#USB mode signal handler
  def handle_usb_mode(self, *args):
    self.__wrappers['usb_mode']['object'].mode_handler(*args)
    self.__check_property_modified(['usb_mode', self.__wrappers['usb_mode']['object'].get_mode()], True)

  @dbus.service.signal(dbus_interface='ru.spirit.advanced_power_monitor.signal', signature='as')
  def PropertyModified(self, keys):
    return keys

  def __check_property_modified(self, *values):
    signals = []
    no_queue = False
    if values:
      values = list(values)
      if type(values[-1]) == bool: no_queue = values.pop(-1)
      for x in values:
        if self.get_value(x[0]) != x[1]:
          self.__set_value(x[0], x[1])
          if len(self.__info_map[x[0]]['clients']): signals.append(x[0])
      if signals:
        if no_queue: self.PropertyModified(signals)
        else: self.__notify(signals)

  def __notify(self, keys):
    watching = []
    for k in keys:
      try:
        clients_len = len(self.__info_map[k]['clients'])
        if clients_len:
          self.__signal_queue.append(k)
          watching.append(clients_len)
      except KeyError:
        pass
    info_map = self.__info_map.values()
    total = len([v for v in info_map if not v['async'] and len(v['clients'])])
    if not total:
      total = 1
      watching = [1]
    c_queue = min(watching, 1 + sum(watching)/(total*len(self.__clients)))
    if c_queue > self.__min_queue: self.__min_queue = c_queue
    result = set(self.__signal_queue)
    if len(result) >= self.__min_queue:
      self.__signal_queue = []
      self.PropertyModified(result)

#Section: Battery state and percentage
#Key: battery_state
  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request')
  def get_status_info(self):
    try: self.__wrappers['battery']['object'].get_status()
    except Exception, e: self.__print_exception('get_status_info', str(e))

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', out_signature='s')
  def get_battery_state(self):
    result = 'unknown'
    if self.battery_is_full:
      result = 'full'
    elif self.battery_is_charging:
      result = 'charging'
    elif self.battery_charger_is_disconnected:
      result = 'charger_disconn'
      self.battery_charger_is_disconnected = False
    else:
      result = '%s%s' % (self.get_value('battery_status'), '%')
    return result
#end

#Section: Bluetooth
#Key: bluetooth_mode
  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', in_signature='s')
  def set_bt_mode(self, mode):
    try:
      if self.__wrappers['bluetooth']['object'] and mode != self.get_value('bluetooth_mode') and mode in self.get_value('bluetooth_modes'):
        if self.get_value('device_mode') == 'flight' and mode != 'off': self.set_device_mode('normal')
        self.__wrappers['bluetooth']['object'].set_mode(mode)
    except Exception, e:
      self.__print_exception('set_bt_mode', str(e))
#end

#Section: MCE
#Key: device_mode
  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request')
  def home_key_pressed_long(self):
    Shell.run_command_async('/usr/bin/dbus-send --system --type=signal /com/nokia/mce/signal com.nokia.mce.signal.sig_home_key_pressed_long_ind')

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request')
  def home_key_pressed(self):
    Shell.run_command_async('/usr/bin/dbus-send --system --type=signal /com/nokia/mce/signal com.nokia.mce.signal.sig_home_key_pressed_ind')

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request')
  def switch_device_mode(self):
    try:
      c_mode = self.get_value('device_mode')
      mode = c_mode
      if c_mode == 'normal':
        Shell.run_command_async('/usr/bin/dbus-send --system --dest=com.nokia.icd /com/nokia/icd_ui com.nokia.icd_ui.disconnect boolean:true')
        mode = 'flight'
      elif c_mode == 'flight':
        mode = 'normal'
      else:
        return
      self.set_device_mode(mode)
    except Exception, e: self.__print_exception('switch_device_mode', str(e))

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', in_signature='s')
  def set_device_mode(self, mode):
    try:
      if mode in ('flight', 'normal') and self.__wrappers['device_mode']['object']:
        self.__wrappers['device_mode']['object'].set_device_mode(mode)
    except Exception, e: self.__print_exception('set_device_mode', str(e))

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request')
  def tklock_device(self):
    try:
      if self.__wrappers['device_mode']['object']: self.__wrappers['device_mode']['object'].tklock()
    except Exception, e: self.__print_exception ('tklock_device', str(e))

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request')
  def reboot(self):
    try:
      if self.__wrappers['device_mode']['object']: self.__wrappers['device_mode']['object'].reboot()
      else: Shell.run_command_async('/sbin/reboot -n')
    except Exception, e: self.__print_exception('reboot', str(e))

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request')
  def shutdown(self):
    try:
      if self.__wrappers['device_mode']['object']: self.__wrappers['device_mode']['object'].shutdown()
      else: Shell.run_command_async('/sbin/shutdown -n')
    except Exception, e: self.__print_exception('shutdown', str(e))
#end

#Section: CPU
#Key: cpu_governor, cpu_frequency
  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', in_signature='s')
  def set_cpu_governor(self, governor):
    if not self.__info_map['cpu_governor']['value'] == governor and governor in self.__info_map['cpu_governors']['value']:
      try:
        self.__interfaces['CPUFreq'].SetCPUFreqGovernor(governor)
        self.__check_property_modified(['cpu_governor', self.__interfaces['CPUFreq'].GetCPUFreqGovernor()])
      except KeyError:
        pass

#Section: USB Mode
#Key: usb_state
  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', in_signature='s')
  def set_usb_mode(self, mode):
    try:
      if self.__wrappers['usb_mode']['object'] and mode != self.get_value('usb_mode') and mode in self.get_value('usb_modes'):
        self.__wrappers['usb_mode']['object'].set_mode(mode)
    except Exception, e:
      self.__print_exception('set_usb_mode', str(e))
#end

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', out_signature='s')
  def get_version(self):
    return self.version

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', out_signature='as')
  def get_property_list(self):
    return self.__info_map.keys()

  @dbus.service.method(dbus_interface='ru.spirit.advanced_power_monitor.request', out_signature='v')
  def get_value(self, key):
    try: return self.__info_map[str(key)]['value']
    except KeyError: return 'Error'

  def __set_value(self, key, value=None):
    try: self.__info_map[str(key)]['value'] = value
    except: pass

  def __add_timer_event(self, key):
    if self.__working and 'time' in self.__info_map[key] and self.__info_map[key]['time'] > 0 and len(self.__info_map[key]['clients']) > 0 and not self.get_value('system_inactive'):
      self.__info_map[key]['src_evt'] = gobject.timeout_add(self.__info_map[key]['time'], self.__update_module, key)

  def __print_exception(self, func_name, message):
    print_exception_message('%s.%s()' % (self.__classname, func_name), message)
