# -*- coding: utf-8 -*-
'''
Advanced Power Common Config
2008-2010(c) Kirill Plyashkevich <ru.spirit@gmail.com>
'''

try: #most distros have a separate gconf
  import gconf
except: #but Maemo doesn't
  from gnome import gconf

class Config():

  def __init__(self, settings_path='/'):
    self.__config = {}
    self.__gconf_path = settings_path
    self.__gconf_client = gconf.client_get_default()
    self.__gconf_types = {gconf.VALUE_LIST: list, gconf.VALUE_PAIR: tuple, gconf.VALUE_STRING: str, gconf.VALUE_INT: int, gconf.VALUE_BOOL: bool, gconf.VALUE_FLOAT: long}
    self.__types = dict(zip(self.__gconf_types.values(), self.__gconf_types.keys()))
    self.__gconf_value_methods = {gconf.VALUE_BOOL: 'bool', gconf.VALUE_INT: 'int', gconf.VALUE_STRING: 'string', gconf.VALUE_FLOAT: 'float', gconf.VALUE_LIST: 'list', gconf.VALUE_SCHEMA: 'schema'}
    self.__options = {}
    self.load()
    #print self.__config

  def get_keys(self):
    return self.__config.keys()

  def get_values(self):
    return self.__config.values()

  def get_option(self, key):
    try:
      return self.__options[key]
    except KeyError:
      return None

  def get_entry(self, key):
    try:
      return self.__config[key]['name'], self.__config[key]['type'], self.__config[key]['value']
    except KeyError:
      return None, None

  def get_value(self, key):
    try:
      return self.__config[key]['value']
    except KeyError:
      print '%s: not found: %s' % (self.__class__, key)
      return None

  def add_entry(self, key_name, key_value=None, save=False, callback=None):
    if not key_name in self.__config:
      self.__config[key_name] = {'type': type(key_value), 'value': key_value, 'name': key_name.split('/')[-1]}
      if save:
        self.save_key(key, callback)
      else:
        if callback:
          callback(new_value)
    return self.__config[key_name]

  def set_value(self, key, new_value, save=False, callback=None):
    try:
      self.__config[key]['value'] = self.__config[key]['type'](new_value)
      if save:
        self.save_key(key, callback)
      else:
        if callback:
          callback(new_value)
    except KeyError:
      self.add_entry(self, key_name, key_value, save, callback)

  def set_options(self, options):
    self.__options = options

  def save(self):
    for key in self.__config:
      self.save_key(key)
    if self.__gconf_client:
      self.__gconf_client.suggest_sync()

  def save_key(self, key, callback=None):
    try:
      self.__set_gvalue('/'.join((self.__gconf_path, key)), self.__config[key]['value'])
      if callback:
        callback(self.__config[key]['value'])
    except KeyError:
      pass

  def load(self):
    if not self.__gconf_client:
      return
    self.__config.clear()
    self.__parse_dir(self.__gconf_path)

  def __parse_dir(self, g_dir):
    gconf_entries = self.__gconf_client.all_entries(g_dir)
    for entry in gconf_entries:
      self.__parse_entry(entry)
    gconf_dirs = self.__gconf_client.all_dirs(g_dir)
    for gconf_dir in gconf_dirs:
      self.__parse_dir(gconf_dir)

  def __parse_entry(self, entry):
    g_path = entry.get_key()
    self.add_entry(g_path.replace(('%s/' % self.__gconf_path), '', 1), self.__get_gvalue(g_path))

  def __get_type(self, g_type):
    try:
      return self.__gconf_types[g_type]
    except KeyError:
      return str

  def __get_gvalue(self, g_path):
    g_value = self.__gconf_client.get(g_path)
    g_type = g_value.type
    value = None
    if g_type == gconf.VALUE_LIST:
      g_list_type = g_value.get_list_type()
      g_list_values = g_value.get_list(g_list_type)
      list_type = self.__get_type(g_list_type)
      value = [self.__get_primitive_gvalue(g_list_value) for g_list_value in g_list_values]
    elif g_type == gconf.VALUE_PAIR:
      pair_car = self.__get_primitive_gvalue(g_value.get_car())
      pair_cdr = self.__get_primitive_gvalue(g_value.get_cdr())
      value = (pair_car, pair_cdr)
    elif g_type == gconf.VALUE_SCHEMA:
      pass
    else:
      value = self.__get_primitive_gvalue(g_value)
    return value

  def __get_primitive_gvalue(self, g_value):
    g_type = g_value.type
    return self.__get_type(g_type)(getattr(g_value, 'get_%s' % (self.__gconf_value_methods[g_type]))())

  def __set_gvalue(self, g_path, value):
    g_value = self.__gconf_client.get(g_path)
    g_type = g_value.type
    kargs = []
    if g_type == gconf.VALUE_LIST:
      kargs = [g_value.get_list_type(), value]
    elif g_type == gconf.VALUE_PAIR:
      kargs = [g_value.get_car().type, g_value.get_cdr().type, value[0], value[1]]
    elif g_type == gconf.VALUE_SCHEMA:
      kargs = [value]
    else:
      kargs = [value]
    getattr(self.__gconf_client, 'set_%s' % (self.__gconf_value_methods[g_type]))(g_path, *kargs)
