# -*- coding: utf-8 -*-

'''
Advanced Interface Switcher
2010-2012(c) Kirill Plyashkevich <ru.spirit@gmail.com>
WLAN interface object
'''

from __future__ import with_statement
import dbus, gobject, threading
from advifswlib.interfaces.Abstract import InterfaceState, AbstractInterface
from advpowcommon.util.execn import *

class WLANInterface(AbstractInterface):

  name = 'WLAN'

  #connection changed
  #0 DBUS_TYPE_STRING              service type or empty string
  #1 DBUS_TYPE_UINT32              service attributes
  #2 DBUS_TYPE_STRING              service id or empty string
  #3 DBUS_TYPE_STRING              network type or empty string
  #4 DBUS_TYPE_UINT32              network attributes
  #5 DBUS_TYPE_ARRAY (BYTE)        network id or empty string (ESSID)
  #6 DBUS_TYPE_STRING              error that occured; empty string on success
  #7 DBUS_TYPE_UINT32              state of the network connection

  #network search
  #0 DBUS_TYPE_STRING              the network type for which search is done
  #1 DBUS_TYPE_UINT32              state of the network connection

  #4 network attributes
  #ICD_NW_ATTR_IAPNAME		0x01000000
  #ICD_NW_ATTR_SILENT		0x02000000
  #ICD_NW_ATTR_AUTOCONNECT	0x04000000
  #ICD_NW_ATTR_SRV_PROVIDER	0x10000000
  #ICD_NW_ATTR_LOCALMASK	0x00FFFFFF

  #7 state of the network connection
  #ICD_STATE_DISCONNECTED		0
  #ICD_STATE_CONNECTING			1
  #ICD_STATE_CONNECTED			2
  #ICD_STATE_DISCONNECTING		3
  #ICD_STATE_LIMITED_CONN_ENABLED	4
  #ICD_STATE_LIMITED_CONN_DISABLED	5
  #ICD_STATE_SEARCH_START		8
  #ICD_STATE_SEARCH_STOP		9
  #ICD_STATE_INTERNAL_ADDRESS_ACQUIRED	15

  #kp51: enormously insane packaging
  #/opt/packet-injection-modules/2.6.28.10-power51/
  #why not just:
  #/opt/lib/modules/2.6.28.10-power51/packet-injection/

  wifi_cmd = {'up': 'sudo /usr/bin/advifsw/wlan_up.sh',
              'down': 'sudo /usr/bin/advifsw/wlan_down.sh',
              'unload': 'sudo /usr/bin/advifsw/wlan_unload.sh',
              'load0': 'sudo /usr/bin/advifsw/wlan_load0.sh',
              'load1': 'sudo /usr/bin/advifsw/wlan_load1.sh',
              'get_driver': "lsmod | grep '^wl12\([x]\{2\}\|51\)\ ' | sed -n s/\ .*//p",
              'check_be': 'sudo /usr/bin/advifsw/get_wlan_path.sh'
             }
  # off: down > unload
  wifi_turn_off_cmd = [wifi_cmd['down'], wifi_cmd['unload']]
  wifi_drivers = ['wl12xx', 'wl1251']

  def pre_init(self):
    self.__lock = threading.Lock()
    self.__strength_timeout = None
    self.__autooff = None
    self.force_startup_mode = True
    self.__has_be = get_shell_command_output(self.wifi_cmd['check_be']) != ''

    self.interfaces = {'Manager': [None,
                                   '/org/freedesktop/Hal/Manager',
                                   'org.freedesktop.Hal.Manager',
                                   None,
                                   False],
                       'ICD2': ['com.nokia.icd2',
                               '/com/nokia/icd2',
                               'com.nokia.icd2',
                               '/usr/sbin/icd2',
                               False],
                       'ICD_UI': ['com.nokia.icd_ui',
                                  '/com/nokia/icd_ui',
                                  'com.nokia.icd_ui',
                                  '/usr/sbin/icd2',
                                  False],
                       'WLANCOND.req': ['com.nokia.wlancond',
                                        '/com/nokia/wlancond/request',
                                        'com.nokia.wlancond.request',
                                        None,
                                        False]}

    self.state = {'name': self.name,
                  'dbus': {'on': {
                             'signal_name': None,
                             'dbus_interface': self.interfaces['Manager'][2],
                             'path': self.interfaces['Manager'][1],
                             'member_keyword': 'member'},
                           'connected': {
                             'signal_name': 'state_sig',
                             'dbus_interface': self.interfaces['ICD2'][2],
                             'path': self.interfaces['ICD2'][1]},
                           'strength': {
                             'signal_name': 'statistics_sig',
                             'dbus_interface': self.interfaces['ICD2'][2],
                             'path': self.interfaces['ICD2'][1]}},
                  'images': ('ifsw_statusarea_wlan_off', 
                             'ifsw_statusarea_wlan_signal_0', 
                             'ifsw_statusarea_wlan_signal_3',
                             'ifsw_statusarea_wlan_connecting',
                             'ifsw_statusarea_wlan_disconnecting'),
                  'images_st': ('ifsw_statusarea_wlan_signal_0', 
                                'ifsw_statusarea_wlan_signal_1', 
                                'ifsw_statusarea_wlan_signal_1', 
                                'ifsw_statusarea_wlan_signal_2', 
                                'ifsw_statusarea_wlan_signal_3', 
                                'ifsw_statusarea_wlan_signal_3'),
                  'text': ['Off', 'On', '<>', 'Connecting', 'Disconnecting'],
                  'show': (True, True, False, False, True),
                  'banner': (True, True, True, True, True),
                  'value': InterfaceState.off,
                  'strength': 0,
                  'modes': ['wlan-stock'],
                  'mode': 0}
    if self.__has_be:
      self.state['modes'].append('wlan-bleeding-edge')

  def post_init(self):
    self.set_config_option('disconnected-timeout', (-1,10,30,60,300))

  def process_signal_on(self, *args, **kwargs):
    result = None
    mode = None
    if args[0] == '/org/freedesktop/Hal/devices/platform_wl12xx':
      if kwargs['member'] == 'DeviceAdded':
        mode = self.__get_mode(self.__get_driver())
        result = InterfaceState.on
        if self.get_config_value('show-connection-dialog'):
          #self.scan_request()
          gobject.idle_add(self.show_connection_dialog)
          #self.show_connection_dialog()
      elif kwargs['member'] == 'DeviceRemoved':
        result = InterfaceState.off
        self.__cancel_autooff()
        self.__cancel_get_strength()
    return result, mode, None

  def process_signal_connected(self, *args):
    result = None
    if len(args) == 8 and args[3].startswith('WLAN_'):
      if args[7] == 2:
        result = InterfaceState.connected
        self.__cancel_autooff()
        if self.get_config_value('show-signal-strength'):
          self.__strength_timeout = gobject.timeout_add(1000, self.get_strength)
        ap_bytename = self.get_interface('WLANCOND.req').status()[0]
        self.network =  ''.join([str(char) for char in ap_bytename])
      elif args[7] == 0 and self.state['value'] != InterfaceState.off: #State: On
        result = InterfaceState.on
        if self.state['value'] == InterfaceState.disconnecting:
          #schedule autoff if needed
          autooff = int(self.get_config_value('disconnected-timeout'))
          if autooff > 0:
            self.__autooff = gobject.timeout_add(autooff*1000, self.turn_off)
      elif args[7] in (1,15) and self.state['value'] != InterfaceState.off:
        #result = InterfaceState.on
        result = InterfaceState.connecting
        self.__cancel_autooff()
      elif args[7] == 3 and self.state['value'] == InterfaceState.connected:
        #result = InterfaceState.on
        result = InterfaceState.disconnecting
        self.__cancel_get_strength()
      else:
        result = self.state['value']
    #print 'args:', args
    return result, None, None

  def process_signal_strength(self, *args):
    return None, None, min(int(args[7]), 5)

  def get_status(self):
    driver = self.__get_driver()
    mode = self.__get_mode(driver)
    if driver == '':
      return InterfaceState.off, mode, None
    else:
      try:
        self.get_interface('ICD2').state_req()
      except:
        self.reinit_interface('ICD2')
      return InterfaceState.on, mode, 0

  def turn_on(self):
    if self.state['value'] == InterfaceState.off:
      #gobject.idle_add(self.run_commands, self.wifi_cmd['load%s' % self.state['mode']], self.wifi_cmd['up'])
      run_async(self.run_commands, [self.wifi_cmd['load%s' % self.state['mode']], self.wifi_cmd['up']])
    elif self.state['value'] == InterfaceState.connected:
      try:
        self.get_interface('ICD2').disconnect_req(dbus.UInt32(0x8000))
      except:
        self.reinit_interface('ICD2')
    return False

  def turn_off(self):
    self.__cancel_autooff()
    if self.state['value'] != InterfaceState.off:
      self.disconnect()
      run_async(self.run_commands, self.wifi_turn_off_cmd)
      #gobject.idle_add(self.run_commands, *self.wifi_turn_off_cmd)
    return False

  def run_commands(self, *cmds):
    with self.__lock:
      run_shell_command(' && '.join(cmds))
      self.switched()
    return False

  def show_connection_dialog(self):
    try:
      self.get_interface('ICD_UI').show_conn_dlg(False)
    except:
      self.reinit_interface('ICD_UI')
      self.get_interface('ICD_UI').show_conn_dlg(False)
    return False

  def scan_request(self):
    try:
      self.get_interface('ICD2').scan_req(dbus.UInt32(0), dbus.Array([], signature='s'))
    except:
      self.reinit_interface('ICD2')

  def get_strength(self):
    if self.state['value'] == InterfaceState.connected and self.get_config_value('show-signal-strength'):
      self.get_interface('ICD2').statistics_req()
      self.__strength_timeout = gobject.timeout_add(5000, self.get_strength)
    return False

  def __cancel_get_strength(self):
    if self.__strength_timeout:
      gobject.source_remove(self.__strength_timeout)
      self.__strength_timeout = None

  def turn_mode(self, mode, startup):
    mode = min(mode, len(self.state['modes']) - 1)
    if self.state['mode'] != mode:
      if self.state['value'] == InterfaceState.off or startup:
        self.state['mode'] = mode
        self.update()
      else:
        self.disconnect()
        # down > unload > load%mode% > up
        #gobject.idle_add(self.run_commands, self.wifi_cmd['down'], self.wifi_cmd['load%s' % mode], self.wifi_cmd['up'])
        run_async(self.run_commands, [self.wifi_cmd['down'], self.wifi_cmd['load%s' % mode], self.wifi_cmd['up']])
    self.switched()

  def __cancel_autooff(self):
    if self.__autooff:
      gobject.source_remove(self.__autooff)
      self.__autooff = None

  def __get_driver(self):
    return get_shell_command_output(self.wifi_cmd['get_driver']).replace('\n', '')

  def __get_mode(self, driver):
    config = 0
    if self.__has_be:
      if driver == '':
        try:
          config = self.state['modes'].index(self.get_config_value('mode-on-startup'))
        except ValueError:
          self.set_config_value('mode-on-startup', self.state['modes'][config], True)
      else:
        config = self.wifi_drivers.index(driver)
    return config

  def disconnect(self):
    if self.state['value'] == InterfaceState.connected:
      try:
        self.get_interface('ICD2').disconnect_req(dbus.UInt32(0x8000))
      except:
        self.reinit_interface('ICD2')

  def config_changed_show_signal_strength(self, old_value, new_value):
    if new_value:
      self.init_signal('strength')
      self.get_strength()
    else:
      self.destroy_signal('strength')
