#
# Copyright 2010 Ruediger Gad <r.c.g@gmx.de>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import cairo
from datetime import datetime
import gconf
import gtk
import math
import os.path
import string

from cairotextwriter import CairoTextWriter
from gtkpixbuftextwriter import GtkPixbufTextWriter

GCONF_DIR = "/apps/Maemo/advanced_clock_plugin/clock"

class Clock():
    def __init__(self, drawing_area, view_id = None):
        self.drawing_area = drawing_area

        self.force_redraw = False
        self.alarm_enabled = False
        self.alarm_time = None
        self.resize_requested = False

        if view_id is None :
            self.gconf_dir = GCONF_DIR
        else :
            self.gconf_dir = GCONF_DIR + "_" + str(view_id)

        self.init_gconf()
        self.load_gconf_settings()

        if self.use_digit_images :
            self.textwriter = GtkPixbufTextWriter(self.text_scaling, "/usr/lib/advanced-clock-plugin/digits/" + self.digit_image_set + "/")
        else :
            self.textwriter = CairoTextWriter(self.text_scaling, self.font_name, self.fill)

    def calculate_text_width(self, text, font_size = 45):
        return self.textwriter.calculate_text_width(text, font_size * self.text_scaling)

    def draw(self):
        self.time = datetime.today()
        self.context = self.drawing_area.window.cairo_create()
        self.context.select_font_face(self.font_name, cairo.FONT_SLANT_NORMAL, cairo.FONT_WEIGHT_NORMAL)

        self.drawable = self.drawing_area.window
        self.fg_color = self.drawable.get_colormap().alloc(int(65535 * self.color_r), int(65535 * self.color_g), int(65535 * self.color_b))
        self.gc = self.drawable.new_gc(foreground = self.fg_color)

        self.textwriter.set_surfaces(self.context, self.drawable, self.gc)

        if self.resize_requested :
            self.resize()
            if self.alarm_enabled :
                self.resize_alarm_enabled()
            self.resize_requested = False

        self.context.set_source_rgba(1.0, 1.0, 1.0, 0.0)
        self.context.set_operator(cairo.OPERATOR_SOURCE)
        self.context.paint()

        self.context.set_source_rgb(self.color_r, self.color_g, self.color_b)

        if self.alarm_enabled and self.alarm_indicator_enabled :
            self.context.save()
            self.draw_alarm_indicator()
            self.context.restore()

        self.context.save()
        self.draw_clock()
        self.context.restore()

    def draw_alarm_indicator(self):
        # Hack to correctly load icon after a restart.
        # Guess is that MyDocs is mounted after the status menu is started.
        # Consequently, the icon is not found and thus not loaded properly even if set correctly.
        if self.alarm_indicator_icon is None and self.alarm_indicator_icon_file != "" and os.path.isfile(self.alarm_indicator_icon_file) :
            self.alarm_indicator_icon = cairo.ImageSurface.create_from_png(self.alarm_indicator_icon_file)

        if self.alarm_indicator_icon is not None :
            self.context.set_source_surface(self.alarm_indicator_icon, self.drawing_area.get_size_request()[0] - self.alarm_indicator_icon.get_width(), 0)
            self.context.paint()
        else:
            self.context.set_source_rgb(1.0, 0.0, 0.0)
            self.context.arc(self.drawing_area.get_size_request()[0] - 8, 8, 7, 0, 2 * math.pi)
            self.context.fill()

    def draw_clock(self):
        print "Implement draw_clock."

    def draw_text(self, text, font_size = 45, x = 0, y = 35):
        self.textwriter.draw_text(text, font_size * self.text_scaling, x, y)

# The following method is currently only a placeholder/reminder 
# and should not be used...
    def gconf_cb(self, client, conn_id, entry, args):
        pass

    def get_alarm_enabled(self):
        return self.alarm_enabled

    def get_alarm_indicator_enabled(self):
        return self.alarm_indicator_enabled

    def get_alarm_time(self):
        return self.alarm_time

    def get_fill(self):
        return self.fill

    def get_font_name(self):
        return self.font_name

    def get_force_redraw(self):
        return self.force_redraw

    def get_gdk_color(self):
        return gtk.gdk.Color(int(self.color_r * 65535), int(self.color_g * 65535), int(self.color_b * 65535))

    def get_show_seconds(self):
        return self.show_seconds

    def get_text_scaling(self):
        return self.text_scaling

    def get_twelve_hour_mode(self):
        return self.twelve_hour_mode

    def init_gconf(self):
        self.gconf_client = gconf.client_get_default()
        if not self.gconf_client.dir_exists(GCONF_DIR) :
            self.gconf_client.set_bool(GCONF_DIR + "/alarm_indicator_enabled", True)

            self.gconf_client.set_bool(GCONF_DIR + "/fill", False)
            self.gconf_client.set_bool(GCONF_DIR + "/show_seconds", True)
  
            self.gconf_client.set_string(GCONF_DIR + "/font_name", "Sans")

            self.gconf_client.set_float(GCONF_DIR + "/color_r", 1.0)
            self.gconf_client.set_float(GCONF_DIR + "/color_g", 1.0)
            self.gconf_client.set_float(GCONF_DIR + "/color_b", 1.0)

            self.gconf_client.set_float(GCONF_DIR + "/text_scaling", 1.0)
            self.gconf_client.set_bool(GCONF_DIR + "/twelve_hour_mode", False)

        if not self.gconf_client.get(GCONF_DIR + "/pad_zeros") :
            self.gconf_client.set_bool(GCONF_DIR + "/pad_zeros", True)
        
        if not self.gconf_client.get(GCONF_DIR + "/time_zone") :
            self.gconf_client.set_string(GCONF_DIR + "/city", "")
            self.gconf_client.set_string(GCONF_DIR + "/country", "")
            self.gconf_client.set_int(GCONF_DIR + "/offset", 0)
            self.gconf_client.set_string(GCONF_DIR + "/time_zone", "")
        
        if not self.gconf_client.get(GCONF_DIR + "/custom_format") :
            self.gconf_client.set_string(GCONF_DIR + "/custom_format", "")

        if not self.gconf_client.get(GCONF_DIR + "/alarm_indicator_icon") :
            self.gconf_client.set_string(GCONF_DIR + "/alarm_indicator_icon", "")

        if not self.gconf_client.get(GCONF_DIR + "/use_digit_images") :
            self.gconf_client.set_bool(GCONF_DIR + "/use_digit_images", False)
            self.gconf_client.set_string(GCONF_DIR + "/digit_image_set", "test-png")

    def load_gconf_settings(self):
        self.alarm_indicator_enabled = self.gconf_client.get_bool(self.gconf_dir + "/alarm_indicator_enabled")
        self.fill = self.gconf_client.get_bool(self.gconf_dir + "/fill")
        self.show_seconds = self.gconf_client.get_bool(self.gconf_dir + "/show_seconds")

        self.font_name = self.gconf_client.get_string(self.gconf_dir + "/font_name")

        self.color_r = self.gconf_client.get_float(self.gconf_dir + "/color_r")
        self.color_g = self.gconf_client.get_float(self.gconf_dir + "/color_g")
        self.color_b = self.gconf_client.get_float(self.gconf_dir + "/color_b")

        self.text_scaling = self.gconf_client.get_float(self.gconf_dir + "/text_scaling")
        self.twelve_hour_mode = self.gconf_client.get_bool(self.gconf_dir + "/twelve_hour_mode")

        self.pad_zeros = self.gconf_client.get_bool(self.gconf_dir + "/pad_zeros")

        self.city = self.gconf_client.get_string(self.gconf_dir + "/city")
        self.country = self.gconf_client.get_string(self.gconf_dir + "/country")
        self.offset = self.gconf_client.get_int(self.gconf_dir + "/offset")
        self.time_zone = self.gconf_client.get_string(self.gconf_dir + "/time_zone")

        self.custom_format = self.gconf_client.get_string(self.gconf_dir + "/custom_format")

        self.alarm_indicator_icon = None
        self.alarm_indicator_icon_file = self.gconf_client.get_string(self.gconf_dir + "/alarm_indicator_icon")
        if os.path.isfile(self.alarm_indicator_icon_file) :
            self.alarm_indicator_icon = cairo.ImageSurface.create_from_png(self.alarm_indicator_icon_file)
        else:
            print "Image file does not exist: " + self.alarm_indicator_icon_file

        self.use_digit_images = self.gconf_client.get_bool(self.gconf_dir + "/use_digit_images")
        self.digit_image_set = self.gconf_client.get_string(self.gconf_dir + "/digit_image_set")

# Currently we do not use the callback (see above).
#        self.gconf_client.notify_add(GCONF_DIR, self.gconf_cb)

    def resize(self):
        print "Implement resize."

    def resize_alarm_enabled(self):
        size = self.drawing_area.get_size_request()
        if self.alarm_indicator_icon is not None :
            self.drawing_area.set_size_request(size[0] + self.alarm_indicator_icon.get_width(), size[1])
        else:
            self.drawing_area.set_size_request(size[0] + 18, size[1])

    def request_resize(self):
        self.resize_requested = True

    def set_alarm_enabled(self, b):
        old_state = self.alarm_enabled
        self.alarm_enabled = b
 
        if (old_state != self.alarm_enabled) and self.alarm_indicator_enabled :
            self.request_resize()

    def set_alarm_indicator_enabled(self, b):
        self.alarm_indicator_enabled = b

    def set_alarm_time(self, time):
        self.alarm_time = time
        
    def set_color(self, r, g, b):
        self.color_r = r
        self.color_g = g
        self.color_b = b

    def set_fill(self, b):
        self.fill = b

    def set_font_name(self, f):
        self.font_name = f

    def set_gdk_color(self, color):
        self.color_r = color.red / 65535.0
        self.color_g = color.green / 65535.0
        self.color_b = color.blue / 65535.0

    def set_show_seconds(self, b):
        self.show_seconds = b

    def set_text_scaling(self, s):
        self.text_scaling = s

    def set_twelve_hour_mode(self, b):
        self.twelve_hour_mode = b

