/*
 *  Copyright 2010 Ruediger Gad
 *
 *  This file is part of Advanced Clock Plugin.
 *
 *  Advanced Clock Plugin is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Advanced Clock Plugin is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Advanced Clock Plugin.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifdef Q_WS_MAEMO_5
#include <cityinfo.h>
#include <clockd/libtime.h>
#endif

#include <pango/pangocairo.h>

#include <QColor>
#include <QDir>
#include <QFile>
#include <QFileDialog>
#include <QList>
#include <QString>

#include "clockdesktopconfigbutton.h"
#include "dialog.h"
#include "ui_dialog.h"

const QString Dialog::GCONF_PATH = "/apps/Maemo/advanced_clock_plugin";
const QString Dialog::GCONF_PATH_CLOCK = Dialog::GCONF_PATH + "/clock";

Dialog::Dialog(int viewId, QWidget *parent) :
    QDialog(parent),
    ui(new Ui::Dialog)
{
    this->viewId = viewId;
    ui->setupUi(this);

    /*
     * Get available clocks.
     */
    QDir clocksDir = QDir("/usr/lib/advanced-clock-plugin/clocks");
    QStringList filter;
    filter.append("*.py");
    QStringList clockFiles = clocksDir.entryList(filter, QDir::Files);

    QStringList clockNames;
    for(int i = 0; i < clockFiles.length(); i++){
        QString file = clockFiles.at(i);
        file.chop(3);
        clockNames.append(file);
    }
    ui->clockStyleComboBox->addItems(clockNames);

    /*
     * Get available fonts.
     */
    PangoFontMap *fontMap;
    fontMap = pango_cairo_font_map_get_default();
    PangoFontFamily **fonts;
    int n_fonts;
    pango_font_map_list_families(fontMap, &fonts, &n_fonts);

    QStringList fontNames;
    for(int i = 0; i < n_fonts; i++){
        fontNames.append(pango_font_family_get_name(fonts[i]));
    }
    ui->fontComboBox->addItems(fontNames);

    /*
     * Free fonts with pango_font_map_free_families ()
     * as described in http://people.redhat.com/otaylor/pango-mirror/api/pango-fonts.html?
     * Problem: pango_font_map_free_families is not declared.
     */

    /*
     * Get available image sets.
     */
    QDir imagesDir = QDir("/usr/lib/advanced-clock-plugin/digits");
    QStringList imageSets = imagesDir.entryList(QDir::Dirs);

    QStringList imageSetNames;
    for(int i = 0; i < imageSets.length(); i++){
        QString dir = imageSets.at(i);
        imageSetNames.append(dir);
    }
    ui->imageSetComboBox->addItems(imageSetNames);

    /*
     * Connect signals.
     */
    connect(ui->colorChooserButton, SIGNAL(colorChanged(QColor)), this, SLOT(updateColor(QColor)));
    connect(ui->colorRSpinBox, SIGNAL(valueChanged(int)), ui->colorChooserButton, SLOT(updateR(int)));
    connect(ui->colorGSpinBox, SIGNAL(valueChanged(int)), ui->colorChooserButton, SLOT(updateG(int)));
    connect(ui->colorBSpinBox, SIGNAL(valueChanged(int)), ui->colorChooserButton, SLOT(updateB(int)));
    connect(ui->alarmIndicatorIconPushButton, SIGNAL(clicked()), this, SLOT(fileDialog()));

#ifdef Q_WS_MAEMO_5
    if(this->viewId < 0){
        gconfClockName = GCONF_PATH + "/clock_name";
        gconfClockPath = Dialog::GCONF_PATH + "/clock";

        clockPerDesktop = new GConfItem(GCONF_PATH + "/clock_per_desktop");
        ui->clockPerDesktopCheckBox->setChecked(clockPerDesktop->value().toBool());

        GConfItem activeViews("/apps/osso/hildon-desktop/views/active");
        QList<QVariant> viewIds = activeViews.value().toList();

        for(int i = 0; i < viewIds.length(); i++){
            ClockDesktopConfigButton *button = new ClockDesktopConfigButton(viewIds.at(i).toInt());
            button->setText("Desktop " + viewIds.at(i).toString());
            ui->clockPerDesktopLayout->addWidget(button);
        }
    }else{
        gconfClockName = GCONF_PATH + "/clock_name_" + QString::number(viewId);
        gconfClockPath = Dialog::GCONF_PATH + "/clock_" + QString::number(viewId);

        ui->clockPerDesktopCheckBox->close();
    }

    /*
     * Initialize GConfItems.
     */
    clockName = new GConfItem(gconfClockName);
    showSeconds = new GConfItem(gconfClockPath + "/show_seconds");
    twelveHourMode = new GConfItem(gconfClockPath + "/twelve_hour_mode");
    fill = new GConfItem(gconfClockPath + "/fill");
    showAlarmIndicator = new GConfItem(gconfClockPath + "/alarm_indicator_enabled");
    colorR = new GConfItem(gconfClockPath + "/color_r");
    colorG = new GConfItem(gconfClockPath + "/color_g");
    colorB = new GConfItem(gconfClockPath + "/color_b");
    fontName = new GConfItem(gconfClockPath + "/font_name");
    textScaling = new GConfItem(gconfClockPath + "/text_scaling");
    padZeros = new GConfItem(gconfClockPath + "/pad_zeros");
    city = new GConfItem(gconfClockPath + "/city");
    country = new GConfItem(gconfClockPath + "/country");
    offset = new GConfItem(gconfClockPath + "/offset");
    timeZone = new GConfItem(gconfClockPath + "/time_zone");
    customFormat = new GConfItem(gconfClockPath + "/custom_format");
    alarmIndicatorIcon = new GConfItem(gconfClockPath + "/alarm_indicator_icon");
    useDigitImages = new GConfItem(gconfClockPath + "/use_digit_images");
    digitImageSet = new GConfItem(gconfClockPath + "/digit_image_set");

    /*
     * Initialize UI with the values stored in gconf.
     */
    ui->clockStyleComboBox->setCurrentIndex(clockNames.indexOf(clockName->value().toString()));
    ui->showSecondsCheckBox->setChecked(showSeconds->value().toBool());
    ui->twelveHourModeCheckBox->setChecked(twelveHourMode->value().toBool());
    ui->fillCheckBox->setChecked(fill->value().toBool());
    ui->showAlarmIndicatorCheckBox->setChecked(showAlarmIndicator->value().toBool());

    QColor color(colorR->value().toFloat() * 255,
                 colorG->value().toFloat() * 255,
                 colorB->value().toFloat() * 255);
    ui->colorChooserButton->setColor(color);
    updateColor(color);

    ui->fontComboBox->setCurrentIndex(fontNames.indexOf(fontName->value().toString()));
    ui->textScalingSpinBox->setValue(textScaling->value().toFloat() * 100);
    ui->padZerosCheckBox->setChecked(padZeros->value().toBool());
    ui->customFormatLineEdit->setText(customFormat->value().toString());

    QString alarmIconName = alarmIndicatorIcon->value().toString();
    if(alarmIconName != NULL
       && alarmIconName.compare(tr("Select File")) != 0
       && QFile::exists(alarmIconName)){
        ui->alarmIndicatorIconLineEdit->setText(alarmIconName);
    }

    ui->imageSetComboBox->setCurrentIndex(imageSetNames.indexOf(digitImageSet->value().toString()));
    ui->digitImageCheckBox->setChecked(useDigitImages->value().toBool());

    /*
     * Get time zones and initialize combo box.
     */
    int tzIndex = 0;
    QStringList timeZoneCities;
    Cityinfo **cities = cityinfo_get_all();
    for (int i = 0; cities && cities[i]; ++i) {
        QString currentName = QString::fromUtf8(cityinfo_get_name(cities[i]));
        QString currentCountry = QString::fromUtf8(cityinfo_get_country(cities[i]));
        timeZoneCities.append(currentCountry + " -- " + currentName);

        cityNames.append(currentName);
        countryNames.append(currentCountry);
        timeZones.append(QString::fromUtf8(cityinfo_get_zone(cities[i])));

        if(currentCountry == country->value().toString()
            && currentName == city->value().toString()){
            tzIndex = i;
        }
    }
    ui->timeZoneComboBox->addItems(timeZoneCities);
    ui->timeZoneComboBox->setCurrentIndex(tzIndex);
    cityinfo_free_all(cities);
#endif
}

Dialog::~Dialog()
{
    delete ui;

#ifdef Q_WS_MAEMO_5
    delete clockName;
    delete showSeconds;
    delete twelveHourMode;
    delete fill;
    delete showAlarmIndicator;
    delete colorR;
    delete colorG;
    delete colorB;
    delete fontName;
    delete textScaling;
    delete padZeros;
    delete city;
    delete country;
    delete offset;
    delete timeZone;
    delete customFormat;
    delete useDigitImages;
    delete digitImageSet;

    if (this->viewId < 0){
        delete clockPerDesktop;
    }
#endif
}

void Dialog::accept(){
#ifdef Q_WS_MAEMO_5
    showSeconds->set(ui->showSecondsCheckBox->isChecked());
    twelveHourMode->set(ui->twelveHourModeCheckBox->isChecked());
    fill->set(ui->fillCheckBox->isChecked());
    showAlarmIndicator->set(ui->showAlarmIndicatorCheckBox->isChecked());

    QColor color = ui->colorChooserButton->getColor();
    int r;
    int g;
    int b;
    color.getRgb(&r, &g, &b);
    colorR->set(r / 255.0);
    colorG->set(g / 255.0);
    colorB->set(b / 255.0);

    fontName->set(ui->fontComboBox->currentText());
    textScaling->set(ui->textScalingSpinBox->value() / 100.0);
    padZeros->set(ui->padZerosCheckBox->isChecked());

    int tzIndex = ui->timeZoneComboBox->currentIndex();
    city->set(cityNames.at(tzIndex));
    country->set(countryNames.at(tzIndex));
    offset->set(time_get_utc_offset(timeZones.at(tzIndex).toUtf8().data()));
    timeZone->set(timeZones.at(tzIndex));

    customFormat->set(ui->customFormatLineEdit->text());

    QString alarmIconName = ui->alarmIndicatorIconLineEdit->text();
    if(alarmIconName != NULL
       && alarmIconName.compare(tr("Select File")) != 0
       && QFile::exists(alarmIconName)){
        alarmIndicatorIcon->set(alarmIconName);
    }else{
        alarmIndicatorIcon->set("");
    }

    if (this->viewId < 0){
        clockPerDesktop->set(ui->clockPerDesktopCheckBox->isChecked());
    }

    digitImageSet->set(ui->imageSetComboBox->currentText());
    useDigitImages->set(ui->digitImageCheckBox->isChecked());

    /*
     * Hack to get things updated even when the clock is not changed.
     * This is currently needed as the settings are currently updated
     * by creating a new clock instance.
     *
     */
    clockName->unset();
    clockName->set(ui->clockStyleComboBox->currentText());
#endif

    done(QDialog::Accepted);
}

void Dialog::fileDialog(){
    qDebug("Entering RecordDialog::fileDialog()...");

    QString selectedFile = ui->alarmIndicatorIconLineEdit->text();
    if(selectedFile == NULL
       || selectedFile.compare(tr("Select File")) == 0
       || (! QFile::exists(selectedFile)) ){
        selectedFile = QDir::home().absolutePath() + "/MyDocs";
    }

    QString newFile = QFileDialog::getOpenFileName(
            this,
            tr("Select Icon."),
            selectedFile,
            tr("PNG Files (*.png)"));

    if(newFile != NULL){
        fileSelected(newFile);
    }
}

void Dialog::fileSelected(QString fileName){
    if(fileName != NULL
       && fileName.compare(tr("Select File")) != 0
       && QFile::exists(fileName)){
        ui->alarmIndicatorIconLineEdit->setText(fileName);
    }
}

void Dialog::updateColor(QColor color){
    int r;
    int g;
    int b;

    color.getRgb(&r, &g, &b);

    ui->colorRSpinBox->setValue(r);
    ui->colorGSpinBox->setValue(g);
    ui->colorBSpinBox->setValue(b);
}
