/*******************************************************************************
 * Copyright (c) 2007-2008 INdT.
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.

 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

/*
 ============================================================================
 Name        : DBusServer.c
 Author      : Joost Kop
 Description : Implements the GLib DBus Server side of the interface
 ============================================================================
 */

#include <glib.h>
#include <dbus/dbus-glib.h>
#include <stdlib.h> /* exit, EXIT_FAILURE */

#include "dbus-common.h"
#include "DBusServer.h"

/* This defines the per-instance state.

   Each GObject must start with the "parent" definition so that common
   operations that all GObjects support can be called on it. */
//typedef struct {...} ActionManagerObject; defined in DBusServer.h

/* Per class state.

   For the first ActionManager implementation we only have the bare minimum,
   that is, the common implementation for any GObject class. */
typedef struct {
  /* The parent class state. */
  GObjectClass parent;
} ActionManagerObjectClass;

/* Forward declaration of the function that will return the GType of
   the ActionManager implementation. Not used in this program since we only
   need to push this over the D-Bus. */
GType actionmanager_object_get_type(void);

/* Macro for the above. It is common to define macros using the
   naming convention (seen below) for all GType implementations,
   and that is why we are going to do that here as well. */
#define VALUE_TYPE_OBJECT (actionmanager_object_get_type())

#define VALUE_OBJECT(object) \
        (G_TYPE_CHECK_INSTANCE_CAST((object), \
         VALUE_TYPE_OBJECT, ActionManagerObject))
#define VALUE_OBJECT_CLASS(klass) \
        (G_TYPE_CHECK_CLASS_CAST((klass), \
         VALUE_TYPE_OBJECT, ActionManagerObjectClass))
#define VALUE_IS_OBJECT(object) \
        (G_TYPE_CHECK_INSTANCE_TYPE((object), \
         VALUE_TYPE_OBJECT))
#define VALUE_IS_OBJECT_CLASS(klass) \
        (G_TYPE_CHECK_CLASS_TYPE((klass), \
         VALUE_TYPE_OBJECT))
#define VALUE_OBJECT_GET_CLASS(obj) \
        (G_TYPE_INSTANCE_GET_CLASS((obj), \
         VALUE_TYPE_OBJECT, ActionManagerObjectClass))

/* Utility macro to define the actionmanager_object GType structure. */
G_DEFINE_TYPE(ActionManagerObject, actionmanager_object, G_TYPE_OBJECT)

/**
 * Since the stub generator will reference the functions from a call
 * table, the functions must be declared before the stub is included.
 */

/**
 * Pull in the stub for the server side.
 */
#include "ActionManager-server-dbus.h"

/* A small macro that will wrap g_print and expand to empty when
   server will daemonize. We use this to add debugging info on
   the server side, but if server will be daemonized, it does not
   make sense to even compile the code in.

   The macro is quite "hairy", but very convenient. */
#ifdef NO_DAEMON
#define dbg(fmtstr, args...) \
  (g_print(PROGNAME ":%s: " fmtstr "\n", __func__, ##args))
#else
#define dbg(dummy...)
#endif

/**
 * Per object initializer
 *
 * Only sets up internal state (both actionmanagers set to zero)
 */
static void actionmanager_object_init(ActionManagerObject* obj) {
  dbg("Called");

  g_assert(obj != NULL);

  obj->pDaemon = 0;
}

/**
 * Per class initializer
 *
 * Registers the type into the GLib/D-Bus wrapper so that it may add
 * its own magic.
 */
static void actionmanager_object_class_init(ActionManagerObjectClass* klass) {

  dbg("Called");

  g_assert(klass != NULL);

  dbg("Binding to GLib/D-Bus");

  /* Time to bind this GType into the GLib/D-Bus wrappers.
     NOTE: This is not yet "publishing" the object on the D-Bus, but
           since it is only allowed to do this once per class
           creation, the safest place to put it is in the class
           initializer.
           Specifically, this function adds "method introspection
           data" to the class so that methods can be called over
           the D-Bus. */
  dbus_g_object_type_install_info(VALUE_TYPE_OBJECT,
                                 &dbus_glib_actionmanager_object_object_info);

  dbg("Done");
  /* All done. Class is ready to be used for instantiating objects */
}

/**
 * Print out an error message and optionally quit (if fatal is TRUE)
 */
#define PROGNAME "ActionManagerDaemon"
static void handleError(const char* msg, const char* reason,
                                         gboolean fatal) {
  g_printerr(PROGNAME ": ERROR: %s (%s)\n", msg, reason);
  if (fatal) {
    exit(EXIT_FAILURE);
  }
}

void ActionManagerDBusServer_Init(struct ActionManagerDaemonC* ActionManager)
{
	  /* The GObject representing a D-Bus connection. */
	  DBusGConnection* bus = NULL;
	  /* Proxy object representing the D-Bus service object. */
	  DBusGProxy* busProxy = NULL;
	  /* Will hold one instance of ActionManagerObject that will serve all the
	     requsts. */
	  static ActionManagerObject* actionmanagerObj = NULL;
	  /* Will store the result of the RequestName RPC here. */
	  guint result;
	  GError* error = NULL;

	  /* Initialize the GType/GObject system. */
	  g_type_init();

	  g_print(PROGNAME ": Connecting to the Session D-Bus.\n");
	  bus = dbus_g_bus_get(DBUS_BUS_SESSION, &error);
	  if (error != NULL) {
	    /* Print error and terminate. */
	    handleError("Could not connect to session bus", error->message, TRUE);
	  }

	  g_print(PROGNAME ": Registering the well-known name (%s)\n",
	          ACTIONMANAGER_SERVICE_NAME);

	  /* In order to register a well-known name, we need to use the
	     "RequestMethod" of the /org/freedesktop/DBus interface. Each
	     bus provides an object that will implement this interface.

	     In order to do the call, we need a proxy object first.
	     DBUS_SERVICE_DBUS = "org.freedesktop.DBus"
	     DBUS_PATH_DBUS = "/org/freedesktop/DBus"
	     DBUS_INTERFACE_DBUS = "org.freedesktop.DBus" */
	  busProxy = dbus_g_proxy_new_for_name(bus,
	                                       DBUS_SERVICE_DBUS,
	                                       DBUS_PATH_DBUS,
	                                       DBUS_INTERFACE_DBUS);
	  if (busProxy == NULL) {
	    handleError("Failed to get a proxy for D-Bus",
	                "Unknown(dbus_g_proxy_new_for_name)", TRUE);
	  }

	  /* Attempt to register the well-known name.
	     The RPC call requires two parameters:
	     - arg0: (D-Bus STRING): name to request
	     - arg1: (D-Bus UINT32): flags for registration.
	       (please see "org.freedesktop.DBus.RequestName" in
	        http://dbus.freedesktop.org/doc/dbus-specification.html)
	     Will return one uint32 giving the result of the RPC call.
	     We are interested in 1 (we are now the primary owner of the name)
	     or 4 (we were already the owner of the name, however in this
	     application it would not make much sense).

	     The function will return FALSE if it sets the GError. */
	  if (!dbus_g_proxy_call(busProxy,
	                         /* Method name to call. */
	                         "RequestName",
	                         /* Where to store the GError. */
	                         &error,
	                         /* Parameter type of argument 0. Note that
	                            since we are dealing with GLib/D-Bus
	                            wrappers, you will need to find a suitable
	                            GType instead of using the "native" D-Bus
	                            type codes. */
	                         G_TYPE_STRING,
	                         /* Data of argument 0. In our case, this is
	                            the well-known name for our server
	                            example ("org.maemo.Platdev_ex"). */
	                         ACTIONMANAGER_SERVICE_NAME,
	                         /* Parameter type of argument 1. */
	                         G_TYPE_UINT,
	                         /* Data of argument 0. This is the "flags"
	                            argument of the "RequestName" method which
	                            can be use to specify what the bus service
	                            should do when the name already exists on
	                            the bus. We will go with defaults. */
	                         0,
	                         /* Input arguments are terminated with a
	                            special GType marker. */
	                         G_TYPE_INVALID,
	                         /* Parameter type of return value 0.
	                            For "RequestName" it is UINT32 so we pick
	                            the GType that maps into UINT32 in the
	                            wrappers. */
	                         G_TYPE_UINT,
	                         /* Data of return value 0. These will always
	                            be pointers to the locations where the
	                            proxy can copy the results. */
	                         &result,
	                         /* Terminate list of return values. */
	                         G_TYPE_INVALID)) {
	    handleError("D-Bus.RequestName RPC failed", error->message,
	                                                TRUE);
	    /* Note that the whole call failed, not "just" the name
	       registration (we deal with that below). This means that
	       something bad probably has happened and there is not much we can
	       do (hence program termination). */
	  }
	  /* Check the result code of the registration RPC. */
	  g_print(PROGNAME ": RequestName returned %d.\n", result);
	  if (result != 1) {
	    handleError("Failed to get the primary well-known name.",
	                "RequestName result != 1", TRUE);
	    /* In this case we could also continue instead of terminating.
	       We could retry the RPC later. Or "lurk" on the bus waiting for
	       someone to tell us what to do. If we would be publishing
	       multiple services and/or interfaces, it even might make sense
	       to continue with the rest anyway.

	       In our simple program, we terminate. Not much left to do for
	       this poor program if the clients will not be able to find the
	       ActionManager object using the well-known name. */
	  }

	  g_print(PROGNAME ": Creating one ActionManager object.\n");
	  /* The NULL at the end means that we have stopped listing the
	     property names and their values that would have been used to
	     set the properties to initial values. Our simple ActionManager
	     implementation does not support GObject properties, and also
	     does not inherit anything interesting from GObject directly, so
	     there are no properties to set. For more examples on properties
	     see the first GTK+ example programs from the maemo Application
	     Development material.

	     NOTE: You need to keep at least one reference to the published
	           object at all times, unless you want it to disappear from
	           the D-Bus (implied by API reference for
	           dbus_g_connection_register_g_object(). */
	  actionmanagerObj = g_object_new(VALUE_TYPE_OBJECT, NULL);
	  actionmanagerObj->pDaemon = ActionManager;
	  if (actionmanagerObj == NULL) {
	    handleError("Failed to create one ActionManager instance.",
	                "Unknown(OOM?)", TRUE);
	  }

	  g_print(PROGNAME ": Registering it on the D-Bus.\n");
	  /* The function does not return any status, so can not check for
	     errors here. */
	  dbus_g_connection_register_g_object(bus,
	                                      ACTIONMANAGER_SERVICE_OBJECT_PATH,
	                                      G_OBJECT(actionmanagerObj));
}
